<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Google_api_clients_model extends CI_model
{

    function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Insert Google API client credentials for a doctor
     * 
     * @param array $data Client credentials data
     * @return int Inserted ID
     */
    function insertGoogleApiClient($data)
    {
        $data['hospital_id'] = $this->session->userdata('hospital_id');
        $this->db->insert('google_api_clients', $data);
        return $this->db->insert_id();
    }

    /**
     * Get Google API client by doctor ID
     * 
     * @param int $doctor_id Doctor ID
     * @return object|null Client data or null if not found
     */
    function getGoogleApiClientByDoctorId($doctor_id)
    {
        $this->db->where('doctor_id', $doctor_id);
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('is_active', 1);
        $query = $this->db->get('google_api_clients');
        return $query->row();
    }

        function getGoogleApiClientByHospitalId()
    {
        $this->db->where('client_name', 'Admin');
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('is_active', 1);
        $query = $this->db->get('google_api_clients');
        return $query->row();
    }

    /**
     * Get all Google API clients for a hospital
     * 
     * @return array List of client data
     */
    function getGoogleApiClients()
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->order_by('id', 'desc');
        $query = $this->db->get('google_api_clients');
        return $query->result();
    }

    /**
     * Update Google API client credentials
     * 
     * @param int $id Client ID
     * @param array $data Updated data
     * @return bool Update success
     */
    function updateGoogleApiClient($id, $data)
    {
        $this->db->where('id', $id);
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        return $this->db->update('google_api_clients', $data);
    }

    /**
     * Delete Google API client
     * 
     * @param int $id Client ID
     * @return bool Delete success
     */
    function deleteGoogleApiClient($id)
    {
        $this->db->where('id', $id);
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        return $this->db->delete('google_api_clients');
    }

    /**
     * Check if a doctor has Google API credentials
     * 
     * @param int $doctor_id Doctor ID
     * @return bool True if credentials exist, false otherwise
     */
    function hasGoogleApiClient($doctor_id)
    {
        $this->db->where('doctor_id', $doctor_id);
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('is_active', 1);
        $count = $this->db->count_all_results('google_api_clients');
        return $count > 0;
    }

    /**
     * Get Google API client by ID
     * 
     * @param int $id Client ID
     * @return object|null Client data or null if not found
     */
    function getGoogleApiClientById($id)
    {
        $this->db->where('id', $id);
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $query = $this->db->get('google_api_clients');
        return $query->row();
    }

    /**
     * Update access token for a Google API client
     * 
     * @param int $id Client ID
     * @param array $token_data Token data including access_token, refresh_token, expires_in, etc.
     * @return bool Update success
     */
    function updateAccessToken($id, $token_data)
    {
        if ($token_data === null) {
            // Clear the token (disconnect)
            $data = array(
                'access_token' => null,
                'refresh_token' => null,
                'token_expires_in' => null,
                'token_created' => null
            );
        } else {
            $data = array(
                'access_token' => json_encode($token_data),
                'token_expires_in' => isset($token_data['expires_in']) ? $token_data['expires_in'] : null,
                'token_created' => time()
            );
        }
        
        $this->db->where('id', $id);
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        return $this->db->update('google_api_clients', $data);
    }

    /**
     * Get access token for a Google API client
     * 
     * @param int $id Client ID
     * @return array|null Token data or null if not found
     */
    function getAccessToken($id)
    {
        $this->db->select('access_token, refresh_token, token_expires_in, token_created');
        $this->db->where('id', $id);
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $query = $this->db->get('google_api_clients');
        $result = $query->row();
        
        if ($result && !empty($result->access_token)) {
            $token_data = json_decode($result->access_token, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                return $token_data;
            }
        }
        
        return null;
    }

    /**
     * Check if access token is expired
     * 
     * @param int $id Client ID
     * @return bool True if expired, false otherwise
     */
    function isTokenExpired($id)
    {
        $this->db->select('token_expires_in, token_created');
        $this->db->where('id', $id);
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $query = $this->db->get('google_api_clients');
        $result = $query->row();
        
        if ($result && $result->token_created && $result->token_expires_in) {
            $expiration_time = $result->token_created + $result->token_expires_in;
            return time() > $expiration_time;
        }
        
        return true; // If we can't determine, assume it's expired
    }
}