<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Google_api_clients extends MX_Controller
{

    function __construct()
    {
        parent::__construct();
        $this->load->model('settings/settings_model');
        $this->load->model('settings/google_api_clients_model');
        $this->load->model('doctor/doctor_model');
        
        // Allow admin, superadmin, and doctors to access this controller
        if (!$this->ion_auth->in_group(array('admin', 'superadmin', 'Doctor'))) {
            redirect('home/permission');
        }
    }

    /**
     * List all Google API clients
     */
    public function index()
    {
        // If user is a doctor, redirect them to their own settings page
        if ($this->ion_auth->in_group('Doctor')) {
            redirect('doctor/google_calendar_settings');
        }
        
        $data['settings'] = $this->settings_model->getSettings();
        $data['google_api_clients'] = $this->google_api_clients_model->getGoogleApiClients();
        $data['doctors'] = $this->doctor_model->getDoctor();
        
        $this->load->view('home/dashboard');
        $this->load->view('settings/google_api_clients', $data);
        $this->load->view('home/footer');
    }

    /**
     * Add new Google API client
     */
    public function add()
    {
        // If user is a doctor, redirect them to their own settings page
        if ($this->ion_auth->in_group('Doctor')) {
            redirect('doctor/google_calendar_settings');
        }
        
        $data = array();
        $data['settings'] = $this->settings_model->getSettings();
        $data['doctors'] = $this->doctor_model->getDoctor();
        
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error">', '</div>');
        
        // Validation rules
        $this->form_validation->set_rules('doctor_id', 'Doctor', 'trim|required|is_natural_no_zero');
        $this->form_validation->set_rules('client_name', 'Client Name', 'trim|required|max_length[255]');
        $this->form_validation->set_rules('client_id', 'Client ID', 'trim|required');
        $this->form_validation->set_rules('client_secret', 'Client Secret', 'trim|required');
        
        if ($this->form_validation->run() == FALSE) {
            $this->load->view('home/dashboard');
            $this->load->view('add_google_api_client', $data);
            $this->load->view('home/footer');
        } else {
            // Process form data
            $doctor_id = $this->input->post('doctor_id');
            
            // Check if doctor already has credentials
            if ($this->google_api_clients_model->hasGoogleApiClient($doctor_id)) {
                $this->session->set_flashdata('error', 'This doctor already has Google API credentials.');
                redirect('settings/google_api_clients/add');
            }
            
            $data = array(
                'doctor_id' => $doctor_id,
                'client_name' => $this->input->post('client_name'),
                'client_id' => $this->input->post('client_id'),
                'client_secret' => $this->input->post('client_secret'),
                'redirect_uris' => $this->input->post('redirect_uris'),
                'auth_provider_x509_cert_url' => $this->input->post('auth_provider_x509_cert_url'),
                'client_x509_cert_url' => $this->input->post('client_x509_cert_url'),
                'auth_uri' => $this->input->post('auth_uri'),
                'token_uri' => $this->input->post('token_uri'),
                'project_id' => $this->input->post('project_id'),
                'is_active' => 1
            );
            
            $insert_id = $this->google_api_clients_model->insertGoogleApiClient($data);
            
            if ($insert_id) {
                $this->session->set_flashdata('success', 'Google API client added successfully.');
                redirect('settings/google_api_clients');
            } else {
                $this->session->set_flashdata('error', 'Failed to add Google API client.');
                redirect('settings/google_api_clients/add');
            }
        }
    }

    /**
     * Edit Google API client
     */
    public function edit()
    {
        $id = $this->input->get('id');
        $data = array();
        $data['settings'] = $this->settings_model->getSettings();
        $data['google_api_client'] = $this->google_api_clients_model->getGoogleApiClientById($id);
        $data['doctors'] = $this->doctor_model->getDoctor();
        
        if (empty($data['google_api_client'])) {
            $this->session->set_flashdata('error', 'Google API client not found.');
            redirect('settings/google_api_clients');
        }
        
        // If user is a doctor, check if they're editing their own credentials
        if ($this->ion_auth->in_group('Doctor')) {
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
            
            if (empty($doctor) || $doctor->id != $data['google_api_client']->doctor_id) {
                $this->session->set_flashdata('error', 'You can only edit your own Google API client settings.');
                redirect('doctor/google_calendar_settings');
            }
        }
        
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error">', '</div>');
        
        // Validation rules
        $this->form_validation->set_rules('client_name', 'Client Name', 'trim|required|max_length[255]');
        $this->form_validation->set_rules('client_id', 'Client ID', 'trim|required');
        
        if ($this->form_validation->run() == FALSE) {
            $this->load->view('home/dashboard');
            $this->load->view('settings/edit_google_api_client', $data);
            $this->load->view('home/footer');
        } else {
            // Process form data
            $data = array(
                'client_name' => $this->input->post('client_name'),
                'client_id' => $this->input->post('client_id'),
                'client_secret' => $this->input->post('client_secret'),
                'redirect_uris' => $this->input->post('redirect_uris'),
                'auth_provider_x509_cert_url' => $this->input->post('auth_provider_x509_cert_url'),
                'client_x509_cert_url' => $this->input->post('client_x509_cert_url'),
                'auth_uri' => $this->input->post('auth_uri'),
                'token_uri' => $this->input->post('token_uri'),
                'project_id' => $this->input->post('project_id'),
                'is_active' => $this->input->post('is_active') ? 1 : 0
            );
            
            $result = $this->google_api_clients_model->updateGoogleApiClient($id, $data);
            
            if ($result) {
                $this->session->set_flashdata('success', 'Google API client updated successfully.');
                if ($this->ion_auth->in_group('Doctor')) {
                    redirect('doctor/google_calendar_settings');
                } else {
                    redirect('settings/google_api_clients');
                }
            } else {
                $this->session->set_flashdata('error', 'Failed to update Google API client.');
                if ($this->ion_auth->in_group('Doctor')) {
                    redirect('doctor/google_calendar_settings/edit');
                } else {
                    redirect('settings/google_api_clients/edit?id=' . $id);
                }
            }
        }
    }

    /**
     * Delete Google API client
     */
    public function delete()
    {
        // Doctors should not be able to delete Google API clients
        if ($this->ion_auth->in_group('Doctor')) {
            $this->session->set_flashdata('error', 'You do not have permission to delete Google API clients.');
            redirect('doctor/google_calendar_settings');
        }
        
        $id = $this->input->get('id');
        $result = $this->google_api_clients_model->deleteGoogleApiClient($id);
        
        if ($result) {
            $this->session->set_flashdata('success', 'Google API client deleted successfully.');
        } else {
            $this->session->set_flashdata('error', 'Failed to delete Google API client.');
        }
        
        redirect('settings/google_api_clients');
    }

    /**
     * View Google API client details
     */
    public function view()
    {
        $id = $this->input->get('id');
        $data = array();
        $data['settings'] = $this->settings_model->getSettings();
        $data['google_api_client'] = $this->google_api_clients_model->getGoogleApiClientById($id);
        
        if (empty($data['google_api_client'])) {
            $this->session->set_flashdata('error', 'Google API client not found.');
            redirect('settings/google_api_clients');
        }
        
        // If user is a doctor, check if they're viewing their own credentials
        if ($this->ion_auth->in_group('Doctor')) {
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
            
            if (empty($doctor) || $doctor->id != $data['google_api_client']->doctor_id) {
                $this->session->set_flashdata('error', 'You can only view your own Google API client settings.');
                redirect('doctor/google_calendar_settings');
            }
        }
        
        $data['doctor'] = $this->doctor_model->getDoctorById($data['google_api_client']->doctor_id);
        
        $this->load->view('home/dashboard');
        $this->load->view('settings/view_google_api_client', $data);
        $this->load->view('home/footer');
    }

    /**
     * Upload client secret JSON file
     */
    public function upload_client_secret()
    {
        $doctor_id = $this->input->post('doctor_id');
        
        if (!empty($_FILES['client_secret_file']['name'])) {
            $config['upload_path'] = './uploads/';
            $config['allowed_types'] = 'json';
            $config['max_size'] = 2048; // 2MB
            $config['file_name'] = 'client_secret_' . $doctor_id . '_' . time() . '.json';
            
            $this->load->library('upload', $config);
            
            if ($this->upload->do_upload('client_secret_file')) {
                $upload_data = $this->upload->data();
                $file_path = $upload_data['full_path'];
                
                // Read and parse the JSON file
                $json_content = file_get_contents($file_path);
                $client_data = json_decode($json_content, true);
                
                if (json_last_error() === JSON_ERROR_NONE && isset($client_data['installed']) || isset($client_data['web'])) {
                    $client_info = isset($client_data['installed']) ? $client_data['installed'] : $client_data['web'];
                    
                    $data = array(
                        'doctor_id' => $doctor_id,
                        'client_name' => isset($client_data['project_id']) ? $client_data['project_id'] : 'Google API Client',
                        'client_id' => isset($client_info['client_id']) ? $client_info['client_id'] : '',
                        'client_secret' => isset($client_info['client_secret']) ? $client_info['client_secret'] : '',
                        'redirect_uris' => isset($client_info['redirect_uris']) ? json_encode($client_info['redirect_uris']) : '',
                        'auth_uri' => isset($client_info['auth_uri']) ? $client_info['auth_uri'] : '',
                        'token_uri' => isset($client_info['token_uri']) ? $client_info['token_uri'] : '',
                        'auth_provider_x509_cert_url' => isset($client_info['auth_provider_x509_cert_url']) ? $client_info['auth_provider_x509_cert_url'] : '',
                        'client_x509_cert_url' => isset($client_info['client_x509_cert_url']) ? $client_info['client_x509_cert_url'] : '',
                        'project_id' => isset($client_data['project_id']) ? $client_data['project_id'] : ''
                    );
                    
                    // Check if doctor already has credentials
                    if ($this->google_api_clients_model->hasGoogleApiClient($doctor_id)) {
                        // Update existing
                        $existing_client = $this->google_api_clients_model->getGoogleApiClientByDoctorId($doctor_id);
                        $this->google_api_clients_model->updateGoogleApiClient($existing_client->id, $data);
                    } else {
                        // Insert new
                        $this->google_api_clients_model->insertGoogleApiClient($data);
                    }
                    
                    // Delete the uploaded file
                    unlink($file_path);
                    
                    echo json_encode(array('status' => 'success', 'message' => 'Client secret uploaded and processed successfully.'));
                } else {
                    echo json_encode(array('status' => 'error', 'message' => 'Invalid JSON file format.'));
                }
            } else {
                echo json_encode(array('status' => 'error', 'message' => $this->upload->display_errors()));
            }
        } else {
            echo json_encode(array('status' => 'error', 'message' => 'No file selected.'));
        }
    }
}