<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Patient_model extends CI_model
{

    function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    function insertPatient($data)
    {
        $data1 = array('hospital_id' => $this->session->userdata('hospital_id'));
        $data2 = array_merge($data, $data1);
        $this->db->insert('patient', $data2);
        
        // Get the inserted patient ID
        $patient_id = $this->db->insert_id();
        
        // Create automatic sharing permissions for group hospitals
        if ($patient_id) {
            $this->createPatientSharingPermissions($patient_id, $this->session->userdata('hospital_id'));
        }
        
        return $patient_id;
    }
    
    // Create sharing permissions for a new patient with all hospitals in same groups
    function createPatientSharingPermissions($patient_id, $owner_hospital_id)
    {
        // For now, just skip creating sharing permissions since they're handled by group membership
        // The patient visibility is controlled by the getPatient* methods checking hospital group membership
        return true;
    }

    function getPatient()
    {
        // Get patients including shared ones from hospital groups
        $hospital_id = $this->session->userdata('hospital_id');
        
        // Check if hospital groups tables exist before using sharing
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            return $this->getPatientWithSharing($hospital_id);
        } else {
            // Fallback to simple implementation if tables don't exist
            $this->db->where('hospital_id', $hospital_id);
        $this->db->order_by('id', 'desc');
        $query = $this->db->get('patient');
            return $query->result();
        }
    }
    
    // Method to get patients including shared ones from hospital groups
    function getPatientWithSharing($hospital_id)
    {
        $this->db->select('p.*, h.name as hospital_name, 
                           (CASE 
                               WHEN p.hospital_id = ' . $hospital_id . ' THEN "own"
                               ELSE "shared"
                           END) as record_type', FALSE);
        $this->db->from('patient p');
        $this->db->join('hospital h', 'p.hospital_id = h.id');
        
        // Get own patients
        $this->db->group_start();
        $this->db->where('p.hospital_id', $hospital_id);
        $this->db->group_end();
        
        // Get shared patients from group hospitals
        $this->db->or_group_start();
        $this->db->where('p.hospital_id IN (
            SELECT DISTINCT hgm2.hospital_id 
            FROM hospital_group_members hgm1 
            JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
            JOIN hospital_groups hg ON hgm1.group_id = hg.id
            WHERE hgm1.hospital_id = ' . $hospital_id . ' 
            AND hgm2.hospital_id != ' . $hospital_id . '
            AND hgm1.status = "active" 
            AND hgm2.status = "active" 
            AND hg.status = "active"
        )', NULL, FALSE);
        $this->db->group_end();
        
        $this->db->order_by('p.id', 'desc');
        $query = $this->db->get();
        return $query->result();
    }

    function getLimit()
    {
        $current = $this->db->get_where('patient', array('hospital_id' => $this->session->userdata('hospital_id')))->num_rows();
        $limit = $this->db->get_where('hospital', array('id' => $this->session->userdata('hospital_id')))->row()->p_limit;
        if (!is_numeric($limit)) {
            $limit = 0;
        }
        return $limit - $current;
    }

    function getPatientWithoutSearch($order, $dir)
    {
        $hospital_id = $this->session->userdata('hospital_id');
        
        // Check if hospital groups tables exist before using sharing
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            // Use sharing functionality
            $this->db->select('p.*, h.name as hospital_name, 
                               (CASE 
                                   WHEN p.hospital_id = ' . $hospital_id . ' THEN "own"
                                   ELSE "shared"
                               END) as record_type', FALSE);
            $this->db->from('patient p');
            $this->db->join('hospital h', 'p.hospital_id = h.id');
            
            // Get own patients
            $this->db->group_start();
            $this->db->where('p.hospital_id', $hospital_id);
            $this->db->group_end();
            
            // Get shared patients from group hospitals
            $this->db->or_group_start();
            $this->db->where('p.hospital_id IN (
                SELECT DISTINCT hgm2.hospital_id 
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ' . $hospital_id . ' 
                AND hgm2.hospital_id != ' . $hospital_id . '
                AND hgm1.status = "active" 
                AND hgm2.status = "active" 
                AND hg.status = "active"
            )', NULL, FALSE);
            $this->db->group_end();
            
            if ($order != null) {
                $this->db->order_by($order, $dir);
            } else {
                $this->db->order_by('p.name', 'asc');
            }
            $query = $this->db->get();
            return $query->result();
        } else {
            // Fallback to simple implementation
        if ($order != null) {
            $this->db->order_by($order, $dir);
        } else {
            $this->db->order_by('name', 'asc');
        }
            $this->db->where('hospital_id', $hospital_id);
        $query = $this->db->get('patient');
        return $query->result();
        }
    }

    function getPatientBySearch($search, $order, $dir)
    {
        $hospital_id = $this->session->userdata('hospital_id');
        
        // Check if hospital groups tables exist before using sharing
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            // Use sharing functionality with improved search
            $this->db->select('p.*, h.name as hospital_name, 
                               (CASE 
                                   WHEN p.hospital_id = ' . $hospital_id . ' THEN "own"
                                   ELSE "shared"
                               END) as record_type', FALSE);
            $this->db->from('patient p');
            $this->db->join('hospital h', 'p.hospital_id = h.id');
            
            // Hospital group filtering
            $this->db->group_start();
            $this->db->where('p.hospital_id', $hospital_id);
            $this->db->or_where('p.hospital_id IN (
                SELECT DISTINCT hgm2.hospital_id 
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ' . $hospital_id . ' 
                AND hgm2.hospital_id != ' . $hospital_id . '
                AND hgm1.status = "active" 
                AND hgm2.status = "active" 
                AND hg.status = "active"
            )', NULL, FALSE);
            $this->db->group_end();
            
            // Search filtering
            $this->db->where("(p.id LIKE '%" . $search . "%' OR p.name LIKE '%" . $search . "%' OR p.phone LIKE '%" . $search . "%' OR p.address LIKE '%" . $search . "%')", NULL, FALSE);
            
            if ($order != null) {
                $this->db->order_by($order, $dir);
            } else {
                $this->db->order_by('p.name', 'asc');
            }
            $query = $this->db->get();
            return $query->result();
        } else {
            // Fallback to simple implementation
        if ($order != null) {
            $this->db->order_by($order, $dir);
        } else {
            $this->db->order_by('name', 'asc');
        }
        $query = $this->db->select('*')
            ->from('patient')
                ->where('hospital_id', $hospital_id)
            ->where("(id LIKE '%" . $search . "%' OR name LIKE '%" . $search . "%' OR phone LIKE '%" . $search . "%' OR address LIKE '%" . $search . "%')", NULL, FALSE)
            ->get();
        return $query->result();
        }
    }

    function getPatientByLimit($limit, $start, $order, $dir)
    {
        $hospital_id = $this->session->userdata('hospital_id');
        
        // Check if hospital groups tables exist before using sharing
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            // Use sharing functionality
            $this->db->select('p.*, h.name as hospital_name, 
                               (CASE 
                                   WHEN p.hospital_id = ' . $hospital_id . ' THEN "own"
                                   ELSE "shared"
                               END) as record_type', FALSE);
            $this->db->from('patient p');
            $this->db->join('hospital h', 'p.hospital_id = h.id');
            
            // Get own patients and shared patients
            $this->db->group_start();
            $this->db->where('p.hospital_id', $hospital_id);
            $this->db->or_where('p.hospital_id IN (
                SELECT DISTINCT hgm2.hospital_id 
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ' . $hospital_id . ' 
                AND hgm2.hospital_id != ' . $hospital_id . '
                AND hgm1.status = "active" 
                AND hgm2.status = "active" 
                AND hg.status = "active"
            )', NULL, FALSE);
            $this->db->group_end();
            
            if ($order != null) {
                $this->db->order_by($order, $dir);
            } else {
                $this->db->order_by('p.name', 'asc');
            }
            $this->db->limit($limit, $start);
            $query = $this->db->get();
            return $query->result();
        } else {
            // Fallback to simple implementation
            $this->db->where('hospital_id', $hospital_id);
        if ($order != null) {
            $this->db->order_by($order, $dir);
        } else {
            $this->db->order_by('name', 'asc');
        }
        $this->db->limit($limit, $start);
        $query = $this->db->get('patient');
        return $query->result();
        }
    }

    function getPatientByLimitBySearch($limit, $start, $search, $order, $dir)
    {
        $hospital_id = $this->session->userdata('hospital_id');
        
        // Check if hospital groups tables exist before using sharing
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            // Use sharing functionality
            $this->db->select('p.*, h.name as hospital_name, 
                               (CASE 
                                   WHEN p.hospital_id = ' . $hospital_id . ' THEN "own"
                                   ELSE "shared"
                               END) as record_type', FALSE);
            $this->db->from('patient p');
            $this->db->join('hospital h', 'p.hospital_id = h.id');
            
            // Get own patients and shared patients
            $this->db->group_start();
            $this->db->where('p.hospital_id', $hospital_id);
            $this->db->or_where('p.hospital_id IN (
                SELECT DISTINCT hgm2.hospital_id 
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ' . $hospital_id . ' 
                AND hgm2.hospital_id != ' . $hospital_id . '
                AND hgm1.status = "active" 
                AND hgm2.status = "active" 
                AND hg.status = "active"
            )', NULL, FALSE);
            $this->db->group_end();
            
            // Search filtering
            $this->db->where("(p.id LIKE '%" . $search . "%' OR p.name LIKE '%" . $search . "%' OR p.phone LIKE '%" . $search . "%' OR p.address LIKE '%" . $search . "%')", NULL, FALSE);
            
            if ($order != null) {
                $this->db->order_by($order, $dir);
            } else {
                $this->db->order_by('p.name', 'asc');
            }
            $this->db->limit($limit, $start);
            $query = $this->db->get();
            return $query->result();
        } else {
            // Fallback to simple implementation
        if ($order != null) {
            $this->db->order_by($order, $dir);
        } else {
            $this->db->order_by('name', 'asc');
        }
        $this->db->limit($limit, $start);
        $query = $this->db->select('*')
            ->from('patient')
                ->where('hospital_id', $hospital_id)
            ->where("(id LIKE '%" . $search . "%' OR name LIKE '%" . $search . "%' OR phone LIKE '%" . $search . "%' OR address LIKE '%" . $search . "%')", NULL, FALSE)
            ->get();
        return $query->result();
        }
    }

    function getPatientById($id)
    {
        // Check if patient is accessible through hospital groups
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            $hospital_id = $this->session->userdata('hospital_id');
            
            // Get patient with hospital information and record type
            $this->db->select('p.*, h.name as hospital_name, 
                               (CASE 
                                   WHEN p.hospital_id = ' . $hospital_id . ' THEN "own"
                                   ELSE "shared"
                               END) as record_type', FALSE);
            $this->db->from('patient p');
            $this->db->join('hospital h', 'p.hospital_id = h.id');
            $this->db->where('p.id', $id);
            
            // Allow access to own patients or patients from group hospitals
            $this->db->group_start();
            $this->db->where('p.hospital_id', $hospital_id);
            $this->db->or_where('p.hospital_id IN (
                SELECT DISTINCT hgm2.hospital_id 
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ' . $hospital_id . ' 
                AND hgm2.hospital_id != ' . $hospital_id . '
                AND hgm1.status = "active" 
                AND hgm2.status = "active" 
                AND hg.status = "active"
            )', NULL, FALSE);
            $this->db->group_end();
            
            $query = $this->db->get();
            return $query->row();
        } else {
            // Fallback to simple implementation if tables don't exist
            $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
            $this->db->where('id', $id);
            $query = $this->db->get('patient');
            return $query->row();
        }
    }

    function getPatientByIonUserId($id)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('ion_user_id', $id);
        $query = $this->db->get('patient');
        return $query->row();
    }

    function getPatientByEmail($email)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('email', $email);
        $query = $this->db->get('patient');
        return $query->row();
    }

    function updatePatient($id, $data)
    {
        $this->db->where('id', $id);
        $this->db->update('patient', $data);
    }

    function delete($id)
    {
        $this->db->where('id', $id);
        $this->db->delete('patient');
    }

    function insertMedicalHistory($data)
    {
        $data1 = array('hospital_id' => $this->session->userdata('hospital_id'));
        $data2 = array_merge($data, $data1);
        $this->db->insert('medical_history', $data2);
    }

    function getMedicalHistoryByPatientId($id)
    {
        $hospital_id = $this->session->userdata('hospital_id');
        
        // Check if hospital groups tables exist before using sharing
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            // Get medical history from all hospitals in the same group, with hospital name
            $this->db->select('mh.*, h.name as hospital_name, 
                               (CASE 
                                   WHEN mh.hospital_id = ' . $hospital_id . ' THEN "own"
                                   ELSE "shared"
                               END) as record_type', FALSE);
            $this->db->from('medical_history mh');
            $this->db->join('hospital h', 'mh.hospital_id = h.id');
            $this->db->where('mh.patient_id', $id);
            
            // Get records from own hospital and shared hospitals in the same group
            $this->db->group_start();
            $this->db->where('mh.hospital_id', $hospital_id);
            $this->db->or_where('mh.hospital_id IN (
                SELECT DISTINCT hgm2.hospital_id 
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ' . $hospital_id . ' 
                AND hgm2.hospital_id != ' . $hospital_id . '
                AND hgm1.status = "active" 
                AND hgm2.status = "active" 
                AND hg.status = "active"
            )', NULL, FALSE);
            $this->db->group_end();
            
            $this->db->order_by('mh.id', 'desc');
            $query = $this->db->get();
            return $query->result();
        } else {
            // Fallback to simple implementation if tables don't exist
            $this->db->where('hospital_id', $hospital_id);
            $this->db->order_by('id', 'desc');
            $this->db->where('patient_id', $id);
            $query = $this->db->get('medical_history');
            return $query->result();
        }
    }

    function getMedicalHistory()
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->order_by('id', 'desc');
        $query = $this->db->get('medical_history');
        return $query->result();
    }

    function getMedicalHistoryBySearch($search)
    {
        $this->db->order_by('id', 'desc');
        $query = $this->db->select('*')
            ->from('medical_history')
            ->where('hospital_id', $this->session->userdata('hospital_id'))
            ->where("(id LIKE '%" . $search . "%' OR patient_name LIKE '%" . $search . "%' OR patient_phone LIKE '%" . $search . "%' OR patient_address LIKE '%" . $search . "%' OR description LIKE '%" . $search . "%')", NULL, FALSE)
            ->get();;
        return $query->result();
    }

    function getMedicalHistoryByLimit($limit, $start)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->order_by('id', 'desc');
        $this->db->limit($limit, $start);
        $query = $this->db->get('medical_history');
        return $query->result();
    }

    function getMedicalHistoryByLimitBySearch($limit, $start, $search)
    {
        $this->db->order_by('id', 'desc');
        $this->db->limit($limit, $start);
        $query = $this->db->select('*')
            ->from('medical_history')
            ->where('hospital_id', $this->session->userdata('hospital_id'))
            ->where("(id LIKE '%" . $search . "%' OR patient_name LIKE '%" . $search . "%' OR patient_phone LIKE '%" . $search . "%' OR patient_address LIKE '%" . $search . "%' OR description LIKE '%" . $search . "%')", NULL, FALSE)
            ->get();;
        return $query->result();
    }

    function getMedicalHistoryById($id)
    {
        $this->db->where('id', $id);
        $query = $this->db->get('medical_history');
        return $query->row();
    }

    function updateMedicalHistory($id, $data)
    {
        $this->db->where('id', $id);
        $this->db->update('medical_history', $data);
    }

    function insertDiagnosticReport($data)
    {
        $data1 = array('hospital_id' => $this->session->userdata('hospital_id'));
        $data2 = array_merge($data, $data1);
        $this->db->insert('diagnostic_report', $data2);
    }

    function updateDiagnosticReport($id, $data)
    {
        $this->db->where('id', $id);
        $this->db->update('diagnostic_report', $data);
    }

    function getDiagnosticReport()
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->order_by('id', 'desc');
        $query = $this->db->get('diagnostic_report');
        return $query->result();
    }

    function getDiagnosticReportById($id)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('id', $id);
        $query = $this->db->get('diagnostic_report');
        return $query->row();
    }

    function getDiagnosticReportByInvoiceId($id)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('invoice', $id);
        $query = $this->db->get('diagnostic_report');
        return $query->row();
    }

    function getDiagnosticReportByPatientId($id)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('patient', $id);
        $query = $this->db->get('diagnostic_report');
        return $query->result();
    }

    function insertPatientMaterial($data)
    {
        $data1 = array('hospital_id' => $this->session->userdata('hospital_id'));
        $data2 = array_merge($data, $data1);
        $this->db->insert('patient_material', $data2);
    }

    function getPatientMaterial()
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->order_by('id', 'desc');
        $query = $this->db->get('patient_material');
        return $query->result();
    }

    function getPatientMaterialWithoutSearch($order, $dir)
    {
        if ($order != null) {
            $this->db->order_by($order, $dir);
        } else {
            $this->db->order_by('id', 'desc');
        }
        $query = $this->db->get('patient_material');
        return $query->result();
    }

    function getDocumentBySearch($search, $order, $dir)
    {
        if ($order != null) {
            $this->db->order_by($order, $dir);
        } else {
            $this->db->order_by('id', 'desc');
        }
        $query = $this->db->select('*')
            ->from('patient_material')
            ->where('hospital_id', $this->session->userdata('hospital_id'))
            ->where("(id LIKE '%" . $search . "%' OR patient_name LIKE '%" . $search . "%' OR patient_phone LIKE '%" . $search . "%' OR patient_address LIKE '%" . $search . "%' OR title LIKE '%" . $search . "%' OR date_string LIKE '%" . $search . "%')", NULL, FALSE)
            ->get();;
        return $query->result();
    }

    function getDocumentByLimit($limit, $start, $order, $dir)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        if ($order != null) {
            $this->db->order_by($order, $dir);
        } else {
            $this->db->order_by('id', 'desc');
        }
        $this->db->limit($limit, $start);
        $query = $this->db->get('patient_material');
        return $query->result();
    }

    function getDocumentByLimitBySearch($limit, $start, $search, $order, $dir)
    {
        if ($order != null) {
            $this->db->order_by($order, $dir);
        } else {
            $this->db->order_by('id', 'desc');
        }
        $this->db->limit($limit, $start);
        $query = $this->db->select('*')
            ->from('patient_material')
            ->where('hospital_id', $this->session->userdata('hospital_id'))
            ->where("(id LIKE '%" . $search . "%' OR patient_name LIKE '%" . $search . "%' OR patient_phone LIKE '%" . $search . "%' OR patient_address LIKE '%" . $search . "%' OR title LIKE '%" . $search . "%' OR date_string LIKE '%" . $search . "%')", NULL, FALSE)
            ->get();;
        return $query->result();
    }

    function getPatientMaterialById($id)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('id', $id);
        $query = $this->db->get('patient_material');
        return $query->row();
    }

    function getPatientMaterialByPatientId($id)
    {
        $hospital_id = $this->session->userdata('hospital_id');
        
        // Check if hospital groups tables exist before using sharing
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            // Get patient materials from all hospitals in the same group, with hospital name
            $this->db->select('pm.*, h.name as hospital_name, 
                               (CASE 
                                   WHEN pm.hospital_id = ' . $hospital_id . ' THEN "own"
                                   ELSE "shared"
                               END) as record_type', FALSE);
            $this->db->from('patient_material pm');
            $this->db->join('hospital h', 'pm.hospital_id = h.id');
            $this->db->where('pm.patient', $id);
            
            // Get records from own hospital and shared hospitals in the same group
            $this->db->group_start();
            $this->db->where('pm.hospital_id', $hospital_id);
            $this->db->or_where('pm.hospital_id IN (
                SELECT DISTINCT hgm2.hospital_id 
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ' . $hospital_id . ' 
                AND hgm2.hospital_id != ' . $hospital_id . '
                AND hgm1.status = "active" 
                AND hgm2.status = "active" 
                AND hg.status = "active"
            )', NULL, FALSE);
            $this->db->group_end();
            
            $this->db->order_by('pm.id', 'desc');
            $query = $this->db->get();
            return $query->result();
        } else {
            // Fallback to simple implementation if tables don't exist
            $this->db->where('hospital_id', $hospital_id);
            $this->db->where('patient', $id);
            $query = $this->db->get('patient_material');
            return $query->result();
        }
    }

    function deletePatientMaterial($id)
    {
        $this->db->where('id', $id);
        $this->db->delete('patient_material');
    }

    function deleteMedicalHistory($id)
    {
        $this->db->where('id', $id);
        $this->db->delete('medical_history');
    }

    function updateIonUser($username, $email, $password, $ion_user_id)
    {
        $uptade_ion_user = array(
            'username' => $username,
            'email' => $email,
            'password' => $password
        );
        $this->db->where('id', $ion_user_id);
        $this->db->update('users', $uptade_ion_user);
    }

    function getDueBalanceByPatientId($patient)
    {
        $query = $this->db->get_where('payment', array('patient' => $patient))->result();
        $deposits = $this->db->get_where('patient_deposit', array('patient' => $patient))->result();
        $balance = array();
        $deposit_balance = array();
        foreach ($query as $gross) {
            $balance[] = $gross->gross_total;
        }
        $balance = array_sum($balance);

        foreach ($deposits as $deposit) {
            $deposit_balance[] = $deposit->deposited_amount;
        }
        $deposit_balance = array_sum($deposit_balance);

        $bill_balance = $balance;

        return $due_balance = $bill_balance - $deposit_balance;
    }

    function getPatientInfoId($searchTerm)
    {
        if (!empty($searchTerm)) {
            $this->db->select('*');
            $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
            $this->db->where("name like '%" . $searchTerm . "%' OR id like '%" . $searchTerm . "%' OR phone like '%" . $searchTerm . "%'");
            $fetched_records = $this->db->get('patient');
            $users = $fetched_records->result_array();
        } else {
            $this->db->select('*');
            $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
            $this->db->limit(10);
            $fetched_records = $this->db->get('patient');
            $users = $fetched_records->result_array();
        }
        // Initialize Array with fetched data
        $data = array();
        foreach ($users as $user) {
            if (empty($user['age'])) {
                $dateOfBirth = $user['birthdate'];
                if (empty($dateOfBirth)) {
                    $age[0] = '0';
                } else {
                    if (strtotime($dateOfBirth)) {
                        $today = date("Y-m-d");
                        $diff = date_diff(date_create($dateOfBirth), date_create($today));
                        $age[0] = $diff->format('%y');
                    } else {
                        $age[0] = '';
                    }
                }
            } else {
                $age = explode('-', $user['age']);
            }
            $data[] = array("id" => $user['id'], "text" => $user['name'] . ' (' . lang('id') . ': ' . $user['id'] . '- ' . lang('phone') . ': ' . $user['phone'] . '- ' . lang('age') . ': ' . $age[0] . ' years )');
        }
        return $data;
    }
    function getPatientInfo($searchTerm)
    {
        if (!empty($searchTerm)) {
            $this->db->select('*');
            $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
            $this->db->where("name like '%" . $searchTerm . "%' OR id like '%" . $searchTerm . "%' OR phone like '%" . $searchTerm . "%'");
            $fetched_records = $this->db->get('patient');
            $users = $fetched_records->result_array();
        } else {
            $this->db->select('*');
            $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
            $this->db->limit(10);
            $fetched_records = $this->db->get('patient');
            $users = $fetched_records->result_array();
        }
        // Initialize Array with fetched data
        $data = array();
        foreach ($users as $user) {
            if (empty($user['age'])) {
                $dateOfBirth = $user['birthdate'];
                if (empty($dateOfBirth)) {
                    $age[0] = '0';
                } else {
                    if (strtotime($dateOfBirth)) {
                        $today = date("Y-m-d");
                        $diff = date_diff(date_create($dateOfBirth), date_create($today));
                        $age[0] = $diff->format('%y');
                    } else {
                        $age[0] = '';
                    }
                }
            } else {
                $age = explode('-', $user['age']);
            }
            $data[] = array("id" => $user['id'], "text" => $user['name'] . ' (' . lang('id') . ': ' . $user['id'] . '- ' . lang('phone') . ': ' . $user['phone'] . '- ' . lang('age') . ': ' . $age[0] . ' years )');
        }
        return $data;
    }

    function getPatientinfoWithAddNewOption($searchTerm)
    {
        if (!empty($searchTerm)) {
            $this->db->select('*');
            $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
            $this->db->where("name like '%" . $searchTerm . "%' OR id like '%" . $searchTerm . "%' OR phone like '%" . $searchTerm . "%'");
            $fetched_records = $this->db->get('patient');
            $users = $fetched_records->result_array();
        } else {
            $this->db->select('*');
            $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
            $this->db->limit(10);
            $fetched_records = $this->db->get('patient');
            $users = $fetched_records->result_array();
        }
        // Initialize Array with fetched data
        $data = array();
        $data[] = array("id" => 'add_new', "text" => lang('add_new'));
        foreach ($users as $user) {

            if (empty($user['age'])) {
                $dateOfBirth = $user['birthdate'];
                if (empty($dateOfBirth)) {
                    $age[0] = '0';
                } else {
                    if (strtotime($dateOfBirth)) {
                        $today = date("Y-m-d");
                        $diff = date_diff(date_create($dateOfBirth), date_create($today));
                        $age[0] = $diff->format('%y');
                    } else {
                        $age[0] = '';
                    }
                }
            } else {
                $age = explode('-', $user['age']);
            }
            $data[] = array("id" => $user['id'], "text" => $user['name'] . ' (' . lang('id') . ': ' . $user['id'] . ' - ' . lang('phone') . ': ' . $user['phone'] . ' - ' . lang('age') . ': ' . $age[0] . ' ' . lang('years') . ')');
        }
        return $data;
    }

    function insertFolder($data)
    {
        $data1 = array('hospital_id' => $this->session->userdata('hospital_id'));
        $data2 = array_merge($data, $data1);
        $this->db->insert('folder', $data2);
    }

    function getFolder()
    {
        $this->db->order_by('id', 'desc');
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $query = $this->db->get('folder');
        return $query->result();
    }

    function getFolderById($id)
    {
        $this->db->where('id', $id);
        $query = $this->db->get('folder');
        return $query->row();
    }

    function updateFolder($id, $data)
    {
        $this->db->where('id', $id);
        $this->db->update('folder', $data);
    }

    function getFolderByPatientId($id)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('patient', $id);
        $query = $this->db->get('folder');
        return $query->result();
    }

    function getPatientMaterialByFolderId($id)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->where('folder', $id);
        $query = $this->db->get('patient_material');
        return $query->result();
    }

    function deleteFolder($id)
    {

        $this->db->where(array('id' => $id));
        $this->db->delete('folder');
    }

    function deletePatientMaterialByFolderId($id)
    {
        $this->db->where('folder', $id);
        $this->db->delete('patient_material');
    }

    public function deleteImage($con)
    {
        // Delete image data 
        $delete = $this->db->delete($this->imgTbl, $con);

        // Return the status 
        return $delete ? true : false;
    }

    function getPatientMaterialByyPatientId($patient_id)
    {
        $this->db->where('hospital_id', $this->session->userdata('hospital_id'));
        $this->db->order_by('id', 'desc');
        $this->db->where('patient', $patient_id);
        $query = $this->db->get('patient_material');
        return $query->result();
    }
    function insertVitalSign($data)
    {
        $data1 = array('hospital_id' => $this->session->userdata('hospital_id'));
        $data2 = array_merge($data, $data1);
        $this->db->insert('vital_signs', $data2);
    }
    function updateVitalSign($id, $data)
    {
        $this->db->where('id', $id);
        $this->db->update('vital_signs', $data);
    }
    function getVitalSignByPatientId($id)
    {
        $hospital_id = $this->session->userdata('hospital_id');
        
        // Check if hospital groups tables exist before using sharing
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            // Get vital signs from all hospitals in the same group, with hospital name
            $this->db->select('vs.*, h.name as hospital_name, 
                               (CASE 
                                   WHEN vs.hospital_id = ' . $hospital_id . ' THEN "own"
                                   ELSE "shared"
                               END) as record_type', FALSE);
            $this->db->from('vital_signs vs');
            $this->db->join('hospital h', 'vs.hospital_id = h.id');
            $this->db->where('vs.patient_id', $id);
            
            // Get records from own hospital and shared hospitals in the same group
            $this->db->group_start();
            $this->db->where('vs.hospital_id', $hospital_id);
            $this->db->or_where('vs.hospital_id IN (
                SELECT DISTINCT hgm2.hospital_id 
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ' . $hospital_id . ' 
                AND hgm2.hospital_id != ' . $hospital_id . '
                AND hgm1.status = "active" 
                AND hgm2.status = "active" 
                AND hg.status = "active"
            )', NULL, FALSE);
            $this->db->group_end();
            
            $this->db->order_by('vs.id', 'desc');
            $query = $this->db->get();
            return $query->result();
        } else {
            // Fallback to simple implementation if tables don't exist
            $this->db->where('hospital_id', $hospital_id);
            $this->db->where('patient_id', $id);
            $query = $this->db->get('vital_signs');
            return $query->result();
        }
    }
    function getVitalSignById($id)
    {
        $this->db->where('id', $id);
        $query = $this->db->get('vital_signs');
        return $query->row();
    }
    function deleteVitalSign($id)
    {
        $this->db->where('id', $id);
        $this->db->delete('vital_signs');
    }
    function getOdontogram($pid)
    {
        $this->db->where('patient_id', $pid);
        $query = $this->db->get('odontogram')->row();

        if (!empty($query)) {
            return $query;
        } else {
            $value = array(
                'patient_id' => $pid, 'Tooth1' => 'white', 'Tooth2' => 'white', 'Tooth3' => 'white',
                'Tooth4' => 'white', 'Tooth5' => 'white', 'Tooth6' => 'white', 'Tooth7' => 'white', 'Tooth8' => 'white', 'Tooth1' => 'white',
                'Tooth9' => 'white', 'Tooth10' => 'white', 'Tooth11' => 'white', 'Tooth12' => 'white',
                'Tooth13' => 'white', 'Tooth14' => 'white', 'Tooth15' => 'white', 'Tooth16' => 'white', 'Tooth17' => 'white',
                'Tooth18' => 'white', 'Tooth19' => 'white', 'Tooth20' => 'white',
                'Tooth21' => 'white', 'Tooth22' => 'white', 'Tooth23' => 'white', 'Tooth24' => 'white',
                'Tooth25' => 'white', 'Tooth26' => 'white', 'Tooth27' => 'white', 'Tooth28' => 'white',
                'Tooth29' => 'white', 'Tooth30' => 'white', 'Tooth31' => 'white', 'Tooth32' => 'white',
                'description' => ''
            );
            $this->db->insert('odontogram', $value);
            $this->db->where('patient_id', $pid);
            $query = $this->db->get('odontogram')->row();
            return $query;
        }
    }
    function odontogram($pid, $value)
    {
        $this->db->where('patient_id', $pid);
        $this->db->update('odontogram', $value);
    }


    public function getConversationHistory($session_id)
    {
        // Ensure that $session_id is safe to use in a query
        $session_id = $this->session->userdata('hospital_id') . '-case-' . $this->db->escape_str($session_id);

        // Query the database for the history using the session ID
        $this->db->select('history');
        $this->db->from('gpt_memory');
        $this->db->where('session_id', $session_id);
        $query = $this->db->get();

        // Check if a history was found
        if ($query->num_rows() > 0) {
            $row = $query->row_array();
            return json_decode($row['history'], true); // Return the history as a PHP array
        } else {
            return []; // Return an empty array if no history was found
        }
    }
    
    // Helper method to check if a patient is accessible by the current hospital
    // (either own patient or shared through hospital groups)
    function isPatientAccessible($patient_id, $hospital_id = null)
    {
        if ($hospital_id === null) {
            $hospital_id = $this->session->userdata('hospital_id');
        }
        
        // First check if patient exists at all (without hospital restriction)
        $this->db->where('id', $patient_id);
        $patient = $this->db->get('patient')->row();
        
        if (!$patient) {
            return false;
        }
        
        // If it's the patient's own hospital, allow access
        if ($patient->hospital_id == $hospital_id) {
            return true;
        }
        
        // Check if hospital groups tables exist and if patient is shared through hospital groups
        if ($this->db->table_exists('hospital_groups') && $this->db->table_exists('hospital_group_members')) {
            // Check if the patient's hospital and current hospital are in the same active group
            $query = $this->db->query("
                SELECT COUNT(*) as count
                FROM hospital_group_members hgm1 
                JOIN hospital_group_members hgm2 ON hgm1.group_id = hgm2.group_id 
                JOIN hospital_groups hg ON hgm1.group_id = hg.id
                WHERE hgm1.hospital_id = ? 
                AND hgm2.hospital_id = ?
                AND hgm1.status = 'active' 
                AND hgm2.status = 'active' 
                AND hg.status = 'active'
            ", array($hospital_id, $patient->hospital_id));
            
            $result = $query->row();
            return $result->count > 0;
        }
        
        return false;
    }
}
