<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Google_auth extends MX_Controller {

    public function __construct() {
        parent::__construct();
        // Don't load the library here anymore, we'll load it with doctor context
    }

    private function get_doctor_id() {
        $doctor_id = $this->input->get('doctor_id');
        
        // If no doctor_id provided, check if user is a doctor
        if (!$doctor_id && $this->ion_auth->in_group('Doctor')) {
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->db->get_where('doctor', array('ion_user_id' => $doctor_ion_id))->row();
            if ($doctor) {
                $doctor_id = $doctor->id;
            }
        }
        
        // If still no doctor_id and user is hospital admin, check for doctor context in session or POST
        if (!$doctor_id && $this->ion_auth->in_group('admin')) {
            // Check if doctor_id is in session (set during admin operations)
            $doctor_id = $this->session->userdata('admin_selected_doctor_id');
            
            // If not in session, check POST data
            if (!$doctor_id) {
                $doctor_id = $this->input->post('doctor_id');
            }
        }
        
        return $doctor_id;
    }

    private function get_user_type_and_id() {
        $user_type = '';
        $user_id = null;
        
        // Check if user is a doctor
        if ($this->ion_auth->in_group('Doctor')) {
            $user_type = 'doctor';
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->db->get_where('doctor', array('ion_user_id' => $doctor_ion_id))->row();
            if ($doctor) {
                $user_id = $doctor->id;
            }
        }
        // Check if user is an admin
        elseif ($this->ion_auth->in_group('admin')) {
            $user_type = 'admin';
            $user_id = $this->ion_auth->get_user_id();
        }
        
        return array('type' => $user_type, 'id' => $user_id);
    }

    public function index() {
        $user_info = $this->get_user_type_and_id();
        
        log_message('info', 'Google Auth Index called with user type: ' . $user_info['type'] . ', user_id: ' . (!empty($user_info['id']) ? $user_info['id'] : 'NOT PROVIDED'));
        
        if (empty($user_info['type']) || empty($user_info['id'])) {
            // Show error - user context required
            log_message('error', 'User context required for Google Calendar authentication.');
            $data['error_message'] = 'User context required for Google Calendar authentication.';
            $this->load->view('google_auth_error_detailed', $data);
            return;
        }
        
        // Load the Google Calendar library with user context
        $this->load->library('google_calendar', array(
            'user_type' => $user_info['type'],
            'user_id' => $user_info['id']
        ));
        
        // Log the credentials being used
        $CI =& get_instance();
        $CI->load->model('settings/google_api_clients_model');
        
        if ($user_info['type'] === 'doctor') {
            $client_credentials = $CI->google_api_clients_model->getGoogleApiClientByDoctorId($user_info['id']);
        } else {
            $client_credentials = $CI->google_api_clients_model->getGoogleApiClientByHospitalId();
        }
        
        if ($client_credentials) {
            log_message('info', ucfirst($user_info['type']) . ' credentials loaded - Client ID: ' . (isset($client_credentials->client_id) ? substr($client_credentials->client_id, 0, 10) . '...' : 'NOT SET'));
            log_message('info', ucfirst($user_info['type']) . ' credentials details - client_id: ' . $client_credentials->client_id . ', client_secret: ' . (!empty($client_credentials->client_secret) ? 'SET' : 'NOT SET'));
        } else {
            log_message('error', 'No credentials found for ' . $user_info['type'] . ' ID: ' . $user_info['id']);
        }
        
        // Check if already authenticated
        if ($this->google_calendar->is_authenticated()) {
            // Redirect to test page if already authenticated
            log_message('info', ucfirst($user_info['type']) . ' already authenticated, redirecting to test page');
            redirect('google_auth/test?' . $user_info['type'] . '_id=' . $user_info['id']);
            return;
        }
        
        // Get auth URL
        $auth_url = $this->google_calendar->get_auth_url();
        log_message('info', 'Generated auth URL: ' . (!empty($auth_url) ? $auth_url : 'FAILED'));
        if (!empty($auth_url)) {
            // Add user info to session so we can retrieve it in callback
            $this->session->set_userdata('google_auth_user_type', $user_info['type']);
            $this->session->set_userdata('google_auth_user_id', $user_info['id']);
            
            // Instead of using redirect(), let's do a manual redirect to avoid URI issues
            log_message('info', 'Redirecting to Google OAuth URL');
            header('Location: ' . $auth_url);
            exit();
        } else {
            // Show error page with more details
            log_message('error', 'Failed to generate Google Calendar authentication URL');
            $data['error_message'] = 'Failed to generate Google Calendar authentication URL. Please check your credentials.';
            $this->load->view('google_auth_error_detailed', $data);
        }
    }

    public function callback() {
        $code = $this->input->get('code');
        $error = $this->input->get('error');
        $error_description = $this->input->get('error_description');
        
        log_message('info', 'Google Auth Callback received - Code: ' . (!empty($code) ? 'Present' : 'Missing') . ', Error: ' . (!empty($error) ? $error : 'None'));
        
        // Log all GET parameters for debugging
        log_message('info', 'All GET parameters: ' . json_encode($this->input->get()));
        log_message('info', 'Raw $_GET: ' . json_encode($_GET));
        log_message('info', 'Raw $_SERVER[QUERY_STRING]: ' . (isset($_SERVER['QUERY_STRING']) ? $_SERVER['QUERY_STRING'] : 'Not set'));
        
        // Retrieve user info from session
        $user_type = $this->session->userdata('google_auth_user_type');
        $user_id = $this->session->userdata('google_auth_user_id');
        
        log_message('info', 'User type from session: ' . (!empty($user_type) ? $user_type : 'Not found'));
        log_message('info', 'User ID from session: ' . (!empty($user_id) ? $user_id : 'Not found'));
        
        // If no user info in session, try to get it from query parameters (fallback)
        if (!$user_type || !$user_id) {
            $state = $this->input->get('state'); // Google might pass state parameter
            if ($state) {
                log_message('info', 'State parameter found: ' . $state);
                // Decode if it's base64 encoded
                $decoded = base64_decode($state, true);
                if ($decoded !== false) {
                    $state_data = json_decode($decoded, true);
                    if (isset($state_data['user_type'])) {
                        $user_type = $state_data['user_type'];
                        log_message('info', 'User type extracted from state: ' . $user_type);
                    }
                    if (isset($state_data['user_id'])) {
                        $user_id = $state_data['user_id'];
                        log_message('info', 'User ID extracted from state: ' . $user_id);
                    }
                }
            }
        }
        
        if (!$user_type || !$user_id) {
            log_message('error', 'User context not found in authentication callback');
            $data['error_message'] = 'User context not found in authentication callback. Please try authenticating again.';
            $this->load->view('google_auth_error_detailed', $data);
            return;
        }
        
        // Check if there's an error from Google
        if (!empty($error)) {
            $error_msg = 'Google returned an error: ' . $error;
            if (!empty($error_description)) {
                $error_msg .= ' - ' . $error_description;
            }
            log_message('error', $error_msg);
            $data['error_message'] = $error_msg;
            $this->load->view('google_auth_error_detailed', $data);
            return;
        }
        
        // Check if we received an authorization code
        if (empty($code)) {
            log_message('error', 'No authorization code received from Google for ' . $user_type . ' ID: ' . $user_id);
            $detailed_error = 'No authorization code received from Google. This could be due to several reasons:<br><br>' .
                           '1. The authentication was cancelled<br>' .
                           '2. There was an issue with the redirect<br>' .
                           '3. Your Google API credentials may be incorrect<br><br>' .
                           'Please try again and ensure your ' . $user_type . '-specific credentials are properly configured.';
            
            // Add more specific debugging information
            if (empty($_GET)) {
                $detailed_error .= '<br><br><strong>Debug Info:</strong> No GET parameters received at all. This suggests the redirect may not be reaching this endpoint correctly.';
            } else {
                $detailed_error .= '<br><br><strong>Debug Info:</strong> Received GET parameters: ' . json_encode(array_keys($_GET));
            }
            
            $data['error_message'] = $detailed_error;
            $this->load->view('google_auth_error_detailed', $data);
            return;
        }
        
        log_message('info', 'Attempting to authenticate ' . $user_type . ' ID: ' . $user_id);
        
        // Load the Google Calendar library with user context
        $this->load->library('google_calendar', array(
            'user_type' => $user_type,
            'user_id' => $user_id
        ));
        
        // Authenticate with the code
        $result = $this->google_calendar->authenticate_with_code($code);
        if ($result) {
            log_message('info', 'Google Calendar authentication successful for ' . $user_type . ' ID: ' . $user_id);
            // Clear the session data
            $this->session->unset_userdata('google_auth_user_type');
            $this->session->unset_userdata('google_auth_user_id');
            
            // Redirect to test page after successful authentication
            $this->session->set_flashdata('success', 'Google Calendar connected successfully!');
            redirect('google_auth/test?' . $user_type . '_id=' . $user_id);
            return;
        } else {
            log_message('error', 'Google Calendar authentication failed for ' . $user_type . ' ID: ' . $user_id);
            // Authentication failed
            $data['error_message'] = 'Google Calendar authentication failed. Please check your ' . $user_type . '-specific credentials and try again.';
            $this->load->view('google_auth_error_detailed', $data);
            return;
        }
    }
    
    public function status() {
        $user_info = $this->get_user_type_and_id();
        
        if ($user_info['type'] && $user_info['id']) {
            $this->load->library('google_calendar', array(
                'user_type' => $user_info['type'],
                'user_id' => $user_info['id']
            ));
            if ($this->google_calendar->is_authenticated()) {
                echo json_encode(['status' => 'connected', 'user_type' => $user_info['type'], 'user_id' => $user_info['id']]);
            } else {
                echo json_encode(['status' => 'disconnected', 'user_type' => $user_info['type'], 'user_id' => $user_info['id']]);
            }
        } else {
            echo json_encode(['status' => 'error', 'message' => 'User context required']);
        }
    }
    
    public function test() {
        $user_info = $this->get_user_type_and_id();
        
        if (!$user_info['type'] || !$user_info['id']) {
            $data['error_message'] = 'User context required for testing.';
            $this->load->view('google_auth_error_detailed', $data);
            return;
        }
        
        $this->load->library('google_calendar', array(
            'user_type' => $user_info['type'],
            'user_id' => $user_info['id']
        ));
        $data['is_connected'] = $this->google_calendar->is_authenticated();
        $data['user_type'] = $user_info['type'];
        $data['user_id'] = $user_info['id'];
        $data['success_message'] = $this->session->flashdata('success');
        $this->load->view('test_calendar', $data);
    }
    
    public function disconnect() {
        $user_info = $this->get_user_type_and_id();
        
        if (!$user_info['type'] || !$user_info['id']) {
            $this->session->set_flashdata('error', 'User context required for disconnecting.');
            redirect('google_auth/test');
            return;
        }
        
        // Load the Google Calendar library with user context
        $this->load->library('google_calendar', array(
            'user_type' => $user_info['type'],
            'user_id' => $user_info['id']
        ));
        
        // Clear the access token from the database
        $this->load->model('settings/google_api_clients_model');
        
        if ($user_info['type'] === 'doctor') {
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByDoctorId($user_info['id']);
        } else {
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByHospitalId();
        }
        
        if ($google_api_client) {
            $this->google_api_clients_model->updateAccessToken($google_api_client->id, null);
        }
        
        $this->session->set_flashdata('success', 'Google Calendar disconnected successfully.');
        redirect('google_auth/test?' . $user_info['type'] . '_id=' . $user_info['id']);
    }
}
