<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Google_calendar_settings extends MX_Controller
{

    function __construct()
    {
        parent::__construct();
        $this->load->model('settings/settings_model');
        $this->load->model('settings/google_api_clients_model');
        $this->load->model('doctor/doctor_model');

        // Only allow doctors to access this controller
        if (!$this->ion_auth->in_group(array('Doctor', 'admin'))) {
            redirect('home/permission');
        }
    }

    /**
     * Display the Google API client settings form for the current doctor
     */
    public function index()
    {
        if ($this->ion_auth->in_group(array('Doctor'))) {
            // Get the current doctor's ID
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);

            if (empty($doctor)) {
                $this->session->set_flashdata('error', 'Doctor not found.');
                redirect('home');
            }

            $doctor_id = $doctor->id;

            // Get existing Google API client settings for this doctor
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByDoctorId($doctor_id);
        } else {
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByHospitalId();
        }
        $data = array();
        $data['settings'] = $this->settings_model->getSettings();
        $data['google_api_client'] = $google_api_client;
        $data['doctor'] = $doctor;

        $this->load->view('home/dashboard');
        $this->load->view('doctor/google_calendar_settings', $data);
        $this->load->view('home/footer');
    }

    /**
     * Display the edit form for Google API client settings
     */
    public function edit()
    {
        if ($this->ion_auth->in_group(array('Doctor'))) {
            // Get the current doctor's ID
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);


            $doctor_id = $doctor->id;

            // Get existing Google API client settings for this doctor
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByDoctorId($doctor_id);
        } else {
            $doctor_id = "Not Doctor";
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByHospitalId();
        }
        if (empty($google_api_client)) {
            $this->session->set_flashdata('error', 'Google API client settings not found.');
            redirect('doctor/google_calendar_settings');
        }

        $data = array();
        $data['settings'] = $this->settings_model->getSettings();
        $data['google_api_client'] = $google_api_client;
        $data['doctor'] = $doctor;

        $this->load->view('home/dashboard');
        $this->load->view('doctor/edit_google_calendar_settings', $data);
        $this->load->view('home/footer');
    }

    /**
     * Save or update Google API client settings for the current doctor
     */
    public function save()
    {
        if ($this->ion_auth->in_group(array('Doctor'))) {
            // Get the current doctor's ID
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);


            $doctor_id = $doctor->id;
        } else {
            $doctor_id = "Not Doctor";
        }
        $this->load->library('form_validation');
        $this->form_validation->set_error_delimiters('<div class="error">', '</div>');

        // Validation rules
        $this->form_validation->set_rules('client_name', 'Client Name', 'trim|required|max_length[255]');
        $this->form_validation->set_rules('client_id', 'Client ID', 'trim|required');
        $this->form_validation->set_rules('client_secret', 'Client Secret', 'trim|required');

        if ($this->form_validation->run() == FALSE) {
            // Validation failed, reload the form with errors
            if ($this->ion_auth->in_group(array('Doctor'))) {
                $google_api_client = $this->google_api_clients_model->getGoogleApiClientByDoctorId($doctor_id);
            } else {
                $google_api_client = $this->google_api_clients_model->getGoogleApiClientByHospitalId();
            }
            $data = array();
            $data['settings'] = $this->settings_model->getSettings();
            $data['google_api_client'] = $google_api_client;
            $data['doctor'] = $doctor;

            // Check if we're editing or adding
            if ($google_api_client) {
                $this->load->view('home/dashboard');
                $this->load->view('doctor/edit_google_calendar_settings', $data);
            } else {
                $this->load->view('home/dashboard');
                $this->load->view('doctor/google_calendar_settings', $data);
            }
            $this->load->view('home/footer');
        } else {
            // Process form data
            $redirect_uris = $this->input->post('redirect_uris');

            // If redirect_uris is provided as JSON string, decode it
            if (!empty($redirect_uris)) {
                $decoded_uris = json_decode($redirect_uris, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded_uris)) {
                    // Keep as JSON string for storage
                } else {
                    // If not valid JSON, treat as comma-separated values
                    $uris = array_map('trim', explode(',', $redirect_uris));
                    $redirect_uris = json_encode($uris);
                }
            } else {
                // Default redirect URIs if none provided
                $redirect_uris = json_encode([base_url('google_auth/callback'), 'http://localhost/google_auth/callback']);
            }

            $data = array(
                'doctor_id' => $doctor_id,
                'client_name' => $this->input->post('client_name'),
                'client_id' => $this->input->post('client_id'),
                'client_secret' => $this->input->post('client_secret'),
                'redirect_uris' => $redirect_uris,
                'auth_provider_x509_cert_url' => $this->input->post('auth_provider_x509_cert_url'),
                'client_x509_cert_url' => $this->input->post('client_x509_cert_url'),
                'auth_uri' => $this->input->post('auth_uri'),
                'token_uri' => $this->input->post('token_uri'),
                'project_id' => $this->input->post('project_id'),
                'is_active' => 1
            );

            // Check if doctor already has credentials
            if ($this->ion_auth->in_group(array('Doctor'))) {
                $existing_client = $this->google_api_clients_model->getGoogleApiClientByDoctorId($doctor_id);
            } else {
                $existing_client = $this->google_api_clients_model->getGoogleApiClientByHospitalId();
            }

            if ($existing_client) {
                // Update existing credentials
                $result = $this->google_api_clients_model->updateGoogleApiClient($existing_client->id, $data);

                if ($result) {
                    $this->session->set_flashdata('success', 'Google API client updated successfully.');
                } else {
                    $this->session->set_flashdata('error', 'Failed to update Google API client.');
                }
                redirect('doctor/google_calendar_settings/edit');
            } else {
                // Insert new credentials
                $insert_id = $this->google_api_clients_model->insertGoogleApiClient($data);

                if ($insert_id) {
                    $this->session->set_flashdata('success', 'Google API client added successfully.');
                } else {
                    $this->session->set_flashdata('error', 'Failed to add Google API client.');
                }
                redirect('doctor/google_calendar_settings');
            }
        }
    }

    /**
     * Display the Google Calendar authentication page for the current doctor
     */
    public function authenticate()
    {
        if ($this->ion_auth->in_group(array('Doctor'))) {
            // Get the current doctor's ID
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);
            $doctor_id = $doctor->id;

            // Get existing Google API client settings for this doctor
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByDoctorId($doctor_id);
        } else {
            $doctor_id = "Not Doctor";
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByHospitalId();
        }
        $is_authenticated = false;
        $calendar_name = '';
        $callback_url = base_url('google_auth/callback');

        if (!empty($google_api_client)) {
            // Load the Google Calendar library to check authentication status
            $this->load->library('google_calendar', $doctor_id);
            $is_authenticated = $this->google_calendar->is_authenticated();

            // If authenticated, try to get calendar info
            if ($is_authenticated && isset($this->google_calendar->service)) {
                try {
                    $calendar = $this->google_calendar->service->calendars->get('primary');
                    if ($calendar && isset($calendar->summary)) {
                        $calendar_name = $calendar->summary;
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting calendar info: ' . $e->getMessage());
                }
            }
        }

        $data = array();
        $data['settings'] = $this->settings_model->getSettings();
        $data['google_api_client'] = $google_api_client;
        $data['doctor'] = $doctor;
        $data['is_authenticated'] = $is_authenticated;
        $data['calendar_name'] = $calendar_name;
        $data['callback_url'] = $callback_url;

        $this->load->view('home/dashboard');
        $this->load->view('doctor/google_calendar_auth', $data);
        $this->load->view('home/footer');
    }

    /**
     * Test Google API connection for the current doctor
     */
    public function test_connection()
    {
        if ($this->ion_auth->in_group(array('Doctor'))) {
            // Get the current doctor's ID
            $doctor_ion_id = $this->ion_auth->get_user_id();
            $doctor = $this->doctor_model->getDoctorByIonUserId($doctor_ion_id);

            $doctor_id = $doctor->id;

            // Check if doctor has Google API credentials
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByDoctorId($doctor_id);
        } else {
            $doctor_id = "Not Doctor";
            $google_api_client = $this->google_api_clients_model->getGoogleApiClientByHospitalId();
        }
        if (empty($google_api_client)) {
            $this->session->set_flashdata('error', 'Google API client settings not found. Please configure your Google API credentials first.');
            redirect('doctor/google_calendar_settings');
        }
        if ($this->ion_auth->in_group(array('Doctor'))) {
            // Load the Google Calendar library
            $this->load->library('google_calendar', $doctor_id);
        } else {
            $this->load->library('google_calendar');
        }
        // Test the connection
        $test_result = $this->google_calendar->test_connection();

        if ($test_result['success']) {
            $this->session->set_flashdata('success', $test_result['message'] . ' ' . $test_result['details']);
        } else {
            $this->session->set_flashdata('error', $test_result['message']);
        }

        // Redirect based on the URI segment
        if ($this->uri->segment(4) == 'edit') {
            redirect('doctor/google_calendar_settings/edit');
        } else {
            redirect('doctor/google_calendar_settings');
        }
    }
}
