<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Google_calendar {

    protected $client;
    protected $service;
    protected $is_loaded = false;
    protected $authenticated = false;
    protected $doctor_id = null;
    protected $user_type = 'doctor'; // Default to doctor, can be 'admin'

    public function __construct($doctor_id = null, $user_type = 'doctor') {
        $this->doctor_id = $doctor_id;
        $this->user_type = $user_type;
        
        // Only try to get doctor info from session if no doctor_id was passed and user is doctor
        if ($this->doctor_id === null && $this->user_type === 'doctor') {
            $this->ion_auth = new Ion_auth();
            $doctor_ion_id = $this->ion_auth->get_user_id();
            
            // Get CI instance to access database
            $CI =& get_instance();
            $doctor = $CI->db->get_where('doctor', array('ion_user_id' => $doctor_ion_id))->row();
            
            if ($doctor) {
                $this->doctor_id = $doctor->id;
            }
        }
        
        $this->load_google_api_client();
    }
    
    private function load_google_api_client() {
        // List of possible autoloader paths
        $autoloadPaths = [
            FCPATH . 'vendor/autoload.php', // Main vendor autoload
            FCPATH . 'vendor/google/apiclient/vendor/autoload.php', // Direct installation
            APPPATH . 'third_party/google_api_php_client/vendor/autoload.php', // Legacy path
            FCPATH . 'application/third_party/google_api_php_client/vendor/autoload.php' // Alternative legacy path
        ];
        
        $autoloadLoaded = false;
        $loadedPath = '';
        foreach ($autoloadPaths as $path) {
            if (file_exists($path)) {
                require_once $path;
                $autoloadLoaded = true;
                $loadedPath = $path;
                break;
            }
        }
        
        if ($autoloadLoaded) {
            log_message('info', 'Google API Client autoloader loaded from: ' . $loadedPath);
            $this->initialize_client();
        } else {
            // Fallback to mock implementation
            log_message('error', 'Google API Client autoloader not found in any expected location. Checked paths: ' . implode(', ', $autoloadPaths));
            $this->setup_mock_implementation();
        }
    }
    
    private function initialize_client() {
        try {
            // Check if Google\Client class exists
            if (!class_exists('Google\\Client') && !class_exists('Google_Client')) {
                log_message('error', 'Google API Client classes not found. Available classes: ' . implode(', ', get_declared_classes()));
                $this->setup_mock_implementation();
                return;
            }
            
            // Use the newer class name if available, otherwise fall back to the older one
            if (class_exists('Google\\Client')) {
                $clientClass = 'Google\\Client';
                log_message('info', 'Using Google\\Client class');
            } else {
                $clientClass = 'Google_Client';
                log_message('info', 'Using Google_Client class');
            }
            
            $this->client = new $clientClass();
            $this->client->setApplicationName('Clinic Appointment Sync');
            
            // Set scopes - use string format to avoid undefined constant errors
            $this->client->setScopes(['https://www.googleapis.com/auth/calendar']);
            
            // Handle different user types
            if ($this->user_type === 'admin') {
                // Admin authentication - use hospital-specific credentials from database
                $CI =& get_instance();
                $CI->load->model('settings/google_api_clients_model');
                $client_credentials = $CI->google_api_clients_model->getGoogleApiClientByHospitalId();
                
                if ($client_credentials) {
                    // Log the credentials being used (without sensitive data)
                    log_message('info', 'Using Google API credentials for admin with client_id: ' . 
                        (isset($client_credentials->client_id) ? substr($client_credentials->client_id, 0, 10) . '...' : 'NOT SET'));
                    
                    // Check if required credentials are present
                    if (empty($client_credentials->client_id) || empty($client_credentials->client_secret)) {
                        log_message('error', 'Missing required Google API credentials for admin: ' . 
                            ' client_id: ' . (!empty($client_credentials->client_id) ? 'SET' : 'MISSING') . 
                            ' client_secret: ' . (!empty($client_credentials->client_secret) ? 'SET' : 'MISSING'));
                        $this->setup_mock_implementation();
                        return;
                    }
                    
                    // Create auth config array from database credentials (web application format)
                    $authConfig = [
                        'web' => [
                            'client_id' => $client_credentials->client_id,
                            'client_secret' => $client_credentials->client_secret,
                            'redirect_uris' => !empty($client_credentials->redirect_uris) ? json_decode($client_credentials->redirect_uris, true) : [base_url('google_auth/callback'), 'http://localhost/google_auth/callback'],
                            'auth_uri' => !empty($client_credentials->auth_uri) ? $client_credentials->auth_uri : 'https://accounts.google.com/o/oauth2/auth',
                            'token_uri' => !empty($client_credentials->token_uri) ? $client_credentials->token_uri : 'https://oauth2.googleapis.com/token',
                        ]
                    ];
                    
                    log_message('info', 'Setting auth config for admin');
                    log_message('info', 'Auth config client_id: ' . substr($authConfig['web']['client_id'], 0, 10) . '...');
                    log_message('info', 'Auth config redirect_uris: ' . json_encode($authConfig['web']['redirect_uris']));
                    
                    $this->client->setAuthConfig($authConfig);
                    
                    // Explicitly set the redirect URI to ensure it matches
                    $this->client->setRedirectUri(base_url('google_auth/callback'));
                    
                    $this->client->setAccessType('offline');
                    $this->client->setPrompt('select_account consent');
                    
                    // Load saved access token for admin
                    if (!empty($client_credentials->access_token)) {
                        $accessToken = json_decode($client_credentials->access_token, true);
                        if (json_last_error() === JSON_ERROR_NONE) {
                            $this->client->setAccessToken($accessToken);
                            $this->authenticated = true;
                            
                            // Log token info (without sensitive data)
                            if (isset($accessToken['expires_in'])) {
                                log_message('info', 'Admin token expires in: ' . $accessToken['expires_in'] . ' seconds');
                            }
                            
                            // If token expired, refresh
                            if ($this->client->isAccessTokenExpired() && $this->client->getRefreshToken()) {
                                log_message('info', 'Refreshing expired access token for admin');
                                $this->client->fetchAccessTokenWithRefreshToken($this->client->getRefreshToken());
                                
                                // Update token in database
                                $newToken = $this->client->getAccessToken();
                                $CI->google_api_clients_model->updateAccessToken($client_credentials->id, $newToken);
                                log_message('info', 'New admin token saved to database');
                            }
                        }
                    }
                } else {
                    log_message('error', 'No Google API credentials found for admin');
                    $this->setup_mock_implementation();
                    return;
                }
            } else {
                // Doctor authentication - use doctor-specific credentials
                if ($this->doctor_id) {
                    // Load doctor-specific credentials from database
                    $CI =& get_instance();
                    $CI->load->model('settings/google_api_clients_model');
                    $client_credentials = $CI->google_api_clients_model->getGoogleApiClientByDoctorId($this->doctor_id);
                    
                    if ($client_credentials) {
                        // Log the credentials being used (without sensitive data)
                        log_message('info', 'Using Google API credentials for doctor ID: ' . $this->doctor_id . 
                            ' with client_id: ' . (isset($client_credentials->client_id) ? substr($client_credentials->client_id, 0, 10) . '...' : 'NOT SET'));
                        
                        // Check if required credentials are present
                        if (empty($client_credentials->client_id) || empty($client_credentials->client_secret)) {
                            log_message('error', 'Missing required Google API credentials for doctor ID: ' . $this->doctor_id . 
                                ' client_id: ' . (!empty($client_credentials->client_id) ? 'SET' : 'MISSING') . 
                                ' client_secret: ' . (!empty($client_credentials->client_secret) ? 'SET' : 'MISSING'));
                            $this->setup_mock_implementation();
                            return;
                        }
                        
                        // Create auth config array from database credentials (web application format)
                        $authConfig = [
                            'web' => [
                                'client_id' => $client_credentials->client_id,
                                'client_secret' => $client_credentials->client_secret,
                                'redirect_uris' => !empty($client_credentials->redirect_uris) ? json_decode($client_credentials->redirect_uris, true) : [base_url('google_auth/callback'), 'http://localhost/google_auth/callback'],
                                'auth_uri' => !empty($client_credentials->auth_uri) ? $client_credentials->auth_uri : 'https://accounts.google.com/o/oauth2/auth',
                                'token_uri' => !empty($client_credentials->token_uri) ? $client_credentials->token_uri : 'https://oauth2.googleapis.com/token',
                            ]
                        ];
                        
                        log_message('info', 'Setting auth config for doctor ID: ' . $this->doctor_id);
                        log_message('info', 'Auth config client_id: ' . substr($authConfig['web']['client_id'], 0, 10) . '...');
                        log_message('info', 'Auth config redirect_uris: ' . json_encode($authConfig['web']['redirect_uris']));
                        
                        $this->client->setAuthConfig($authConfig);
                        
                        // Explicitly set the redirect URI to ensure it matches
                        $this->client->setRedirectUri(base_url('google_auth/callback'));
                        
                        // Load saved access token for this doctor
                        if (!empty($client_credentials->access_token)) {
                            $accessToken = json_decode($client_credentials->access_token, true);
                            if (json_last_error() === JSON_ERROR_NONE) {
                                $this->client->setAccessToken($accessToken);
                                $this->authenticated = true;
                                
                                // Log token info (without sensitive data)
                                if (isset($accessToken['expires_in'])) {
                                    log_message('info', 'Token expires in: ' . $accessToken['expires_in'] . ' seconds');
                                }
                                
                                // If token expired, refresh
                                if ($this->client->isAccessTokenExpired() && $this->client->getRefreshToken()) {
                                    log_message('info', 'Refreshing expired access token for doctor ID: ' . $this->doctor_id);
                                    $this->client->fetchAccessTokenWithRefreshToken($this->client->getRefreshToken());
                                    
                                    // Update token in database
                                    $newToken = $this->client->getAccessToken();
                                    $CI->google_api_clients_model->updateAccessToken($client_credentials->id, $newToken);
                                    log_message('info', 'New token saved to database for doctor ID: ' . $this->doctor_id);
                                }
                            }
                        }
                    } else {
                        log_message('error', 'No Google API credentials found for doctor ID: ' . $this->doctor_id);
                        $this->setup_mock_implementation();
                        return;
                    }
                } else {
                    // No doctor ID provided for doctor authentication
                    log_message('error', 'Doctor ID required for doctor authentication');
                    $this->setup_mock_implementation();
                    return;
                }
            }

            // Try to initialize the Calendar service
            if (class_exists('Google\\Service\\Calendar')) {
                $calendarClass = 'Google\\Service\\Calendar';
                log_message('info', 'Using Google\\Service\\Calendar class');
                $this->service = new $calendarClass($this->client);
                $this->is_loaded = true;
            } else if (class_exists('Google_Service_Calendar')) {
                $calendarClass = 'Google_Service_Calendar';
                log_message('info', 'Using Google_Service_Calendar class');
                $this->service = new $calendarClass($this->client);
                $this->is_loaded = true;
            } else {
                log_message('info', 'Google Calendar service not available, using mock implementation');
                $this->setup_mock_implementation();
            }
        } catch (Exception $e) {
            log_message('error', 'Google Calendar initialization error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
            $this->setup_mock_implementation();
        }
    }
    
    private function setup_mock_implementation() {
        // Check if we have credentials
        $tokenPath = APPPATH . 'credentials/token.json';
        if (file_exists($tokenPath)) {
            $this->authenticated = true;
        }
        
        $this->is_loaded = true;
    }

    public function add_event($title, $desc, $start_datetime, $end_datetime, $timezone = 'Asia/Dhaka', $attendees = array()) {
        // If using mock implementation
        if (!$this->is_loaded || !isset($this->service)) {
            log_message('info', 'Using mock implementation for Google Calendar event creation');
            return $this->mock_add_event($title, $desc, $start_datetime, $end_datetime, $timezone);
        }
        
        try {
            log_message('info', 'Creating Google Calendar event: ' . $title . ' from ' . $start_datetime . ' to ' . $end_datetime);
            
            // Check if Calendar Event class exists - try newer format first, then fall back
            if (class_exists('Google\Service\Calendar\Event')) {
                $eventClass = 'Google\Service\Calendar\Event';
                log_message('info', 'Using Google\Service\Calendar\Event class');
            } else if (class_exists('Google_Service_Calendar_Event')) {
                $eventClass = 'Google_Service_Calendar_Event';
                log_message('info', 'Using Google_Service_Calendar_Event class');
            } else {
                // Fallback to mock implementation
                log_message('info', 'Event class not found, using mock implementation');
                return $this->mock_add_event($title, $desc, $start_datetime, $end_datetime, $timezone);
            }
            
            // Check if the datetime strings already include timezone information (UTC 'Z' suffix)
            $is_utc = substr($start_datetime, -1) === 'Z' && substr($end_datetime, -1) === 'Z';
            
            // Prepare event data
            $eventData = array(
                'summary' => $title,
                'description' => $desc,
                'start' => array(
                    'dateTime' => $start_datetime,
                ),
                'end' => array(
                    'dateTime' => $end_datetime,
                ),
            );
            
            // Add timezone if not UTC
            if (!$is_utc) {
                $eventData['start']['timeZone'] = $timezone;
                $eventData['end']['timeZone'] = $timezone;
            }
            
            // Add attendees if provided
            if (!empty($attendees)) {
                $eventData['attendees'] = array();
                foreach ($attendees as $attendee) {
                    $eventData['attendees'][] = array('email' => $attendee);
                }
            }
            
            $event = new $eventClass($eventData);

            $event = $this->service->events->insert('primary', $event);
            $eventId = $event->getId();
            log_message('info', 'Google Calendar event created successfully with ID: ' . $eventId);
            return $eventId;
        } catch (Exception $e) {
            log_message('error', 'Google Calendar API Error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
            return false;
        }
    }
    
    private function mock_add_event($title, $desc, $start_datetime, $end_datetime, $timezone = 'Asia/Dhaka') {
        // Mock implementation - just log the event
        $event_data = array(
            'summary' => $title,
            'description' => $desc,
            'start' => array('dateTime' => $start_datetime, 'timeZone' => $timezone),
            'end' => array('dateTime' => $end_datetime, 'timeZone' => $timezone),
        );
        
        log_message('info', 'Google Calendar Event (Mock): ' . json_encode($event_data));
        
        // In a real implementation, this would return the actual event ID
        return 'mock_event_id_' . time();
    }
    
    public function is_authenticated() {
        return $this->authenticated;
    }
    
    public function is_loaded() {
        return $this->is_loaded;
    }
    
    public function get_auth_url() {
        // If using mock implementation
        if (!$this->is_loaded || !isset($this->client)) {
            log_message('error', 'Google Calendar client not loaded properly for ' . $this->user_type . 
                ($this->doctor_id ? ' ID: ' . $this->doctor_id : ''));
            // Return a mock auth URL for now
            return base_url('google_auth/callback');
        }
        
        try {
            // Ensure redirect URI is set
            $redirectUri = base_url('google_auth/callback');
            $this->client->setRedirectUri($redirectUri);
            
            // Set access type to offline to get refresh token
            $this->client->setAccessType('offline');
            
            // Set prompt to ensure we get consent
            $this->client->setPrompt('consent');
            
            // Add state parameter to pass doctor_id only for doctors
            if ($this->user_type === 'doctor' && $this->doctor_id) {
                $state = base64_encode(json_encode(['doctor_id' => $this->doctor_id, 'user_type' => 'doctor']));
                $this->client->setState($state);
                log_message('info', 'State parameter set for doctor ID: ' . $this->doctor_id);
            } else if ($this->user_type === 'admin') {
                $state = base64_encode(json_encode(['user_type' => 'admin']));
                $this->client->setState($state);
                log_message('info', 'State parameter set for admin user');
            }
            
            // Log the client configuration
            log_message('info', 'Creating auth URL for ' . $this->user_type . 
                ($this->doctor_id ? ' ID ' . $this->doctor_id : ''));
            
            $auth_url = $this->client->createAuthUrl();
            log_message('info', 'Generated auth URL for ' . $this->user_type . 
                ($this->doctor_id ? ' ID ' . $this->doctor_id : '') . ': ' . $auth_url);
            log_message('info', 'Redirect URI set to: ' . $redirectUri);
            return $auth_url;
        } catch (Exception $e) {
            log_message('error', 'Google Calendar Auth URL Error for ' . $this->user_type . 
                ($this->doctor_id ? ' ID ' . $this->doctor_id : '') . ': ' . $e->getMessage());
            return base_url('google_auth/callback'); // Fallback
        }
    }
    
    public function authenticate_with_code($code) {
        // If using mock implementation
        if (!$this->is_loaded || !isset($this->client)) {
            log_message('error', 'Google Calendar library not loaded properly for ' . $this->user_type . 
                ($this->doctor_id ? ' ID: ' . $this->doctor_id : ''));
            return false;
        }
        
        try {
            log_message('info', 'Attempting to authenticate with code for ' . $this->user_type . 
                ($this->doctor_id ? ' ID: ' . $this->doctor_id : ''));
            log_message('info', 'Authorization code received: ' . (!empty($code) ? 'Present' : 'Missing'));
            
            $token = $this->client->fetchAccessTokenWithAuthCode($code);
            
            // Log the received token (without sensitive parts)
            log_message('info', 'Received token for ' . $this->user_type . 
                ($this->doctor_id ? ' ID ' . $this->doctor_id : '') . ': ' . 
                (isset($token['access_token']) ? 'access_token present' : 'no access_token') . ', ' .
                (isset($token['expires_in']) ? 'expires_in: ' . $token['expires_in'] : 'no expires_in'));
            
            // Check if we got an error in the token response
            if (isset($token['error'])) {
                log_message('error', 'Google API returned error during token fetch for ' . $this->user_type . 
                    ($this->doctor_id ? ' ID ' . $this->doctor_id : '') . ': ' . $token['error']);
                if (isset($token['error_description'])) {
                    log_message('error', 'Error description: ' . $token['error_description']);
                }
                return false;
            }
            
            // Handle different user types for token saving
            if ($this->user_type === 'admin') {
                // Save token to database for admin
                $CI =& get_instance();
                $CI->load->model('settings/google_api_clients_model');
                $client_credentials = $CI->google_api_clients_model->getGoogleApiClientByHospitalId();
                
                if ($client_credentials) {
                    // Update token in database
                    $CI->google_api_clients_model->updateAccessToken($client_credentials->id, $token);
                    $this->authenticated = true;
                    log_message('info', 'Successfully saved admin token to database');
                    return true;
                } else {
                    log_message('error', 'No Google API client found for admin');
                    return false;
                }
            } else if ($this->user_type === 'doctor' && $this->doctor_id) {
                // Save token to database for doctor
                $CI =& get_instance();
                $CI->load->model('settings/google_api_clients_model');
                $client_credentials = $CI->google_api_clients_model->getGoogleApiClientByDoctorId($this->doctor_id);
                
                if ($client_credentials) {
                    // Update token in database
                    $CI->google_api_clients_model->updateAccessToken($client_credentials->id, $token);
                    $this->authenticated = true;
                    log_message('info', 'Successfully saved token to database for doctor ID: ' . $this->doctor_id);
                    return true;
                } else {
                    log_message('error', 'No Google API client found for doctor ID: ' . $this->doctor_id);
                    return false;
                }
            } else {
                log_message('error', 'Invalid user type or missing doctor ID for token saving');
                return false;
            }
        } catch (Exception $e) {
            log_message('error', 'Google Calendar Auth Error for ' . $this->user_type . 
                ($this->doctor_id ? ' ID ' . $this->doctor_id : '') . ': ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete an event from Google Calendar
     * 
     * @param string $event_id The Google Calendar event ID
     * @return bool True on success, false on failure
     */
    public function delete_event($event_id) {
        // If using mock implementation
        if (!$this->is_loaded || !isset($this->service)) {
            return $this->mock_delete_event($event_id);
        }
        
        try {
            log_message('info', 'Deleting Google Calendar event with ID: ' . $event_id);
            
            // Check if Calendar Event class exists - try newer format first, then fall back
            if (class_exists('Google\Service\Calendar\Event') || class_exists('Google_Service_Calendar_Event')) {
                $this->service->events->delete('primary', $event_id);
                log_message('info', 'Google Calendar event deleted successfully');
                return true;
            } else {
                // Fallback to mock implementation
                log_message('info', 'Event class not found, using mock implementation for delete');
                return $this->mock_delete_event($event_id);
            }
        } catch (Exception $e) {
            log_message('error', 'Google Calendar API Delete Error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
            return false;
        }
    }
    
    private function mock_delete_event($event_id) {
        // Mock implementation - just log the deletion
        log_message('info', 'Google Calendar Event Deletion (Mock): Event ID ' . $event_id);
        
        // In a real implementation, this would actually delete the event
        return true;
    }
    
    /**
     * Test the Google API connection
     * 
     * @return array Test result with status and message
     */
    public function test_connection() {
        $result = array(
            'success' => false,
            'message' => '',
            'details' => ''
        );
        
        // Check if library is loaded
        if (!$this->is_loaded) {
            $result['message'] = 'Google API Client library is not loaded.';
            return $result;
        }
        
        // Check if we have a client object
        if (!isset($this->client) || !$this->client) {
            $result['message'] = 'Google API Client is not initialized.';
            return $result;
        }
        
        try {
            // Check if authenticated
            if (!$this->is_authenticated()) {
                $result['message'] = 'Google API authentication failed. Please check your credentials.';
                return $result;
            }
            
            // Check if we have a service object
            if (!isset($this->service) || !$this->service) {
                $result['message'] = 'Google Calendar service is not available.';
                return $result;
            }
            
            // Try to get the primary calendar to test the connection
            $calendar = $this->service->calendars->get('primary');
            
            if ($calendar && isset($calendar->id)) {
                $result['success'] = true;
                $result['message'] = 'Google API connection test successful!';
                $result['details'] = 'Connected to calendar: ' . (isset($calendar->summary) ? $calendar->summary : $calendar->id);
            } else {
                $result['message'] = 'Unable to retrieve primary calendar information.';
            }
        } catch (Exception $e) {
            $result['message'] = 'Google API connection test failed: ' . $e->getMessage();
            log_message('error', 'Google Calendar Test Connection Error: ' . $e->getMessage());
        }
        
        return $result;
    }
}