<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Google_calendar_debug extends MX_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->library('google_calendar');
    }

    public function index() {
        echo "<h1>Google Calendar Integration Debug</h1>";
        
        // Check if Google API client is installed
        $this->check_google_api_client();
        
        // Check credentials files
        $this->check_credentials();
        
        // Check authentication status
        $this->check_authentication();
        
        // Test event creation (mock)
        $this->test_event_creation();
        
        echo "<h2>Next Steps</h2>";
        echo "<ol>";
        echo "<li>Make sure you have a valid <code>client_secret.json</code> file in <code>application/credentials/</code></li>";
        echo "<li>Visit <a href='" . base_url('google_auth') . "'>Google Auth</a> to authenticate with Google</li>";
        echo "<li>After authentication, create a new appointment to test the integration</li>";
        echo "</ol>";
    }
    
    private function check_google_api_client() {
        echo "<h2>Google API Client Check</h2>";
        
        // Check various autoload paths
        $autoloadPaths = [
            FCPATH . 'vendor/autoload.php',
            FCPATH . 'vendor/google/apiclient/vendor/autoload.php',
            APPPATH . 'third_party/google_api_php_client/vendor/autoload.php',
            FCPATH . 'application/third_party/google_api_php_client/vendor/autoload.php'
        ];
        
        $autoloadFound = false;
        foreach ($autoloadPaths as $path) {
            if (file_exists($path)) {
                echo "<p>✅ Autoload found: " . htmlspecialchars($path) . "</p>";
                $autoloadFound = true;
            }
        }
        
        if (!$autoloadFound) {
            echo "<p>❌ No autoload file found in expected locations</p>";
        }
        
        // Check if Google classes exist
        $classes = [
            'Google\Client' => class_exists('Google\Client'),
            'Google_Client' => class_exists('Google_Client'),
            'Google\Service\Calendar' => class_exists('Google\Service\Calendar'),
            'Google_Service_Calendar' => class_exists('Google_Service_Calendar')
        ];
        
        echo "<h3>Required Classes:</h3>";
        foreach ($classes as $class => $exists) {
            echo "<p>" . ($exists ? "✅" : "❌") . " " . htmlspecialchars($class) . "</p>";
        }
    }
    
    private function check_credentials() {
        echo "<h2>Credentials Check</h2>";
        
        $credentialsDir = APPPATH . 'credentials/';
        echo "<p>Credentials directory: " . htmlspecialchars($credentialsDir) . "</p>";
        
        if (is_dir($credentialsDir)) {
            $files = scandir($credentialsDir);
            foreach ($files as $file) {
                if ($file !== '.' && $file !== '..') {
                    $filePath = $credentialsDir . $file;
                    $fileSize = filesize($filePath);
                    echo "<p>📄 " . htmlspecialchars($file) . " (" . $fileSize . " bytes)</p>";
                }
            }
        } else {
            echo "<p>❌ Credentials directory not found</p>";
        }
        
        // Check specific files
        $clientSecret = $credentialsDir . 'client_secret.json';
        $tokenFile = $credentialsDir . 'token.json';
        
        echo "<h3>Required Files:</h3>";
        echo "<p>" . (file_exists($clientSecret) ? "✅" : "❌") . " client_secret.json</p>";
        echo "<p>" . (file_exists($tokenFile) ? "✅" : "❌") . " token.json (required for authentication)</p>";
    }
    
    private function check_authentication() {
        echo "<h2>Authentication Check</h2>";
        
        $isAuthenticated = $this->google_calendar->is_authenticated();
        echo "<p>Authentication Status: " . ($isAuthenticated ? "✅ Authenticated" : "❌ Not Authenticated") . "</p>";
        
        if (!$isAuthenticated) {
            echo "<p>To authenticate:</p>";
            echo "<ol>";
            echo "<li>Visit <a href='" . base_url('google_auth') . "'>Google Auth</a></li>";
            echo "<li>Sign in with your Google account</li>";
            echo "<li>Grant the necessary permissions</li>";
            echo "<li>You should be redirected back to the test page</li>";
            echo "</ol>";
        }
    }
    
    private function test_event_creation() {
        echo "<h2>Event Creation Test (Mock)</h2>";
        
        // Test creating a mock event
        $event_id = $this->google_calendar->add_event(
            'Test Event',
            'This is a test event for debugging',
            date('c'),
            date('c', time() + 3600)
        );
        
        if ($event_id) {
            echo "<p>✅ Event creation test successful. Event ID: " . htmlspecialchars($event_id) . "</p>";
            if (strpos($event_id, 'mock_event_id_') === 0) {
                echo "<p>⚠️ This is a mock event. Real Google Calendar integration requires authentication.</p>";
            }
        } else {
            echo "<p>❌ Event creation test failed</p>";
        }
    }
    
    public function authenticate() {
        // Redirect to Google Auth
        redirect('google_auth');
    }
}