<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class PatientSelection extends MX_Controller
{
    function __construct()
    {
        parent::__construct();
        $this->load->model('patient/patient_model');
        
        // Ensure user is logged in
        if (!$this->ion_auth->logged_in()) {
            redirect('auth/login');
        }
    }

    /**
     * Display patient selection page for guardians
     */
    public function index()
    {
        // Only allow Patient group users to access this
        if (!$this->ion_auth->in_group('Patient')) {
            redirect('home/permission');
        }

        $user_id = $this->ion_auth->get_user_id();
        
        // Get guardian's patient record
        $guardian_patient = $this->patient_model->getPatientByIonUserId($user_id);
        
        if (!$guardian_patient || $guardian_patient->is_dependent) {
            // Not a guardian, redirect to regular dashboard
            redirect('patient/dashboard');
        }

        // Get all accessible patients (guardian + dependents)
        $accessible_patients = $this->patient_model->getAccessiblePatients($user_id);
        
        if (count($accessible_patients) <= 1) {
            // Only guardian themselves, no dependents - go directly to dashboard
            $this->session->set_userdata('active_patient_id', $guardian_patient->id);
            redirect('patient/dashboard');
        }

        // Prepare data for the view
        $data = array();
        $data['guardian'] = $guardian_patient;
        $data['accessible_patients'] = $accessible_patients;
        $data['current_patient_id'] = $this->session->userdata('active_patient_id');
        
        // Organize patients by type
        $data['guardian_data'] = null;
        $data['dependents'] = array();
        
        foreach ($accessible_patients as $patient) {
            if (!$patient->is_dependent) {
                $data['guardian_data'] = $patient;
            } else {
                $data['dependents'][] = $patient;
            }
        }

        $this->load->view('home/dashboard');
        $this->load->view('patient_selection', $data);
        $this->load->view('home/footer');
    }

    /**
     * Switch to a specific patient context
     */
    public function switchToPatient($patient_id)
    {
        // Only allow Patient group users
        if (!$this->ion_auth->in_group('Patient')) {
            show_error('Access denied');
            return;
        }

        $user_id = $this->ion_auth->get_user_id();
        
        // Verify the guardian can access this patient
        if (!$this->patient_model->isPatientAccessible($user_id, $patient_id)) {
            show_error('Access denied - You cannot access this patient');
            return;
        }

        // Set active patient context
        $this->session->set_userdata('active_patient_id', $patient_id);
        
        // Get patient data for success message
        $patient = $this->patient_model->getPatientById($patient_id);
        $message = "Switched to " . $patient->name;
        if ($patient->is_dependent) {
            $message .= " (" . ucfirst($patient->relationship) . ")";
        } else {
            $message .= " (Your Account)";
        }
        
        show_swal($message, 'success', 'Switched');
        redirect('patient/dashboard');
    }

    /**
     * Get current active patient context
     */
    public function getCurrentPatient()
    {
        $user_id = $this->ion_auth->get_user_id();
        $active_patient_id = $this->session->userdata('active_patient_id');
        
        if (empty($active_patient_id)) {
            // No active patient set, use guardian's own record
            $guardian_patient = $this->patient_model->getPatientByIonUserId($user_id);
            if ($guardian_patient) {
                $this->session->set_userdata('active_patient_id', $guardian_patient->id);
                return $guardian_patient;
            }
            return null;
        }
        
        // Verify access to the active patient
        if (!$this->patient_model->isPatientAccessible($user_id, $active_patient_id)) {
            // Access denied, reset to guardian's own record
            $guardian_patient = $this->patient_model->getPatientByIonUserId($user_id);
            if ($guardian_patient) {
                $this->session->set_userdata('active_patient_id', $guardian_patient->id);
                return $guardian_patient;
            }
            return null;
        }
        
        return $this->patient_model->getPatientById($active_patient_id);
    }

    /**
     * API endpoint to get accessible patients (for AJAX)
     */
    public function getAccessiblePatientsJson()
    {
        if (!$this->ion_auth->in_group('Patient')) {
            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode(['error' => 'Access denied']));
            return;
        }

        $user_id = $this->ion_auth->get_user_id();
        $patients = $this->patient_model->getAccessiblePatients($user_id);
        
        // Format for JSON response
        $formatted_patients = array();
        foreach ($patients as $patient) {
            $age_parts = explode('-', $patient->age);
            $formatted_patients[] = array(
                'id' => $patient->id,
                'name' => $patient->name,
                'age' => $age_parts[0] ?? '0',
                'sex' => $patient->sex,
                'relationship' => $patient->is_dependent ? ucfirst($patient->relationship) : 'Self',
                'is_dependent' => $patient->is_dependent,
                'phone' => $patient->phone,
                'display_name' => $patient->is_dependent 
                    ? $patient->name . ' (' . ucfirst($patient->relationship) . ')'
                    : $patient->name . ' (Your Account)'
            );
        }

        $this->output
            ->set_content_type('application/json')
            ->set_output(json_encode([
                'success' => true,
                'patients' => $formatted_patients,
                'active_patient_id' => $this->session->userdata('active_patient_id')
            ]));
    }

    /**
     * Quick switch widget for patient selection (can be embedded in other pages)
     */
    public function quickSwitchWidget()
    {
        if (!$this->ion_auth->in_group('Patient')) {
            return;
        }

        $user_id = $this->ion_auth->get_user_id();
        $accessible_patients = $this->patient_model->getAccessiblePatients($user_id);
        
        if (count($accessible_patients) <= 1) {
            return; // No widget needed if only one patient
        }

        $data = array();
        $data['accessible_patients'] = $accessible_patients;
        $data['current_patient'] = $this->getCurrentPatient();
        
        $this->load->view('patient_quick_switch_widget', $data);
    }
}
?> 