<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Doctor Registration - <?php echo $settings->title; ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#3B82F6',
                        secondary: '#64748B'
                    }
                }
            }
        }
    </script>
    <style>
        .step { display: none; }
        .step.active { display: block; }
        .pulse-animation { animation: pulse 1.5s infinite; }
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        .hospital-card {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        .hospital-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }
        .hospital-card.selected {
            border-color: #3B82F6;
            background-color: #EBF8FF;
        }
        
        /* Additional custom styles for modal animations */
        .modal-overlay {
            backdrop-filter: blur(2px);
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-lg">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="<?php echo base_url('frontend/public_portal'); ?>" class="text-2xl font-bold text-primary">
                        <i class="fas fa-user-md mr-2"></i><?php echo $settings->title; ?>
                    </a>
                </div>
                <div class="hidden md:flex items-center space-x-8">
                    <a href="<?php echo base_url('frontend/public_portal'); ?>" class="text-gray-700 hover:text-primary transition-colors">Home</a>
                    <a href="<?php echo base_url('frontend/patient_registration'); ?>" class="text-gray-700 hover:text-primary transition-colors">Patient Registration</a>
                    <a href="<?php echo base_url('frontend/appointment_request'); ?>" class="text-gray-700 hover:text-primary transition-colors">Book Appointment</a>
                </div>
                <button id="mobile-menu-button" class="md:hidden flex items-center text-gray-700">
                    <i class="fas fa-bars text-xl"></i>
                </button>
            </div>
        </div>
        <!-- Mobile menu -->
        <div id="mobile-menu" class="md:hidden hidden bg-white border-t">
            <div class="px-2 pt-2 pb-3 space-y-1">
                <a href="<?php echo base_url('frontend/public_portal'); ?>" class="block px-3 py-2 text-gray-700">Home</a>
                <a href="<?php echo base_url('frontend/patient_registration'); ?>" class="block px-3 py-2 text-gray-700">Patient Registration</a>
                <a href="<?php echo base_url('frontend/appointment_request'); ?>" class="block px-3 py-2 text-gray-700">Book Appointment</a>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <section class="py-16 bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
        <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="text-center mb-12">
                <h1 class="text-4xl font-bold text-gray-900 mb-4">Doctor Registration</h1>
                <p class="text-xl text-gray-600">Join our healthcare network in 3 simple steps</p>
            </div>

            <div class="bg-white rounded-2xl shadow-xl overflow-hidden">
                <div class="p-8">
                    <div class="registration-container">
                        <!-- Step Indicator -->
                        <div class="flex justify-center items-center mb-8">
                            <div class="flex items-center space-x-8">
                                <div class="flex items-center space-x-2">
                                    <div class="step-circle w-8 h-8 bg-primary text-white rounded-full flex items-center justify-center text-sm font-medium" id="step-indicator-1">1</div>
                                    <span class="text-sm font-medium text-gray-600">Check</span>
                                </div>
                                <div class="w-16 h-1 bg-gray-200 rounded-full">
                                    <div class="progress-bar w-0 h-full bg-primary rounded-full transition-all duration-500" id="progress-1"></div>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <div class="step-circle w-8 h-8 bg-gray-200 text-gray-400 rounded-full flex items-center justify-center text-sm font-medium" id="step-indicator-2">2</div>
                                    <span class="text-sm font-medium text-gray-400">Hospital</span>
                                </div>
                                <div class="w-16 h-1 bg-gray-200 rounded-full">
                                    <div class="progress-bar w-0 h-full bg-primary rounded-full transition-all duration-500" id="progress-2"></div>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <div class="step-circle w-8 h-8 bg-gray-200 text-gray-400 rounded-full flex items-center justify-center text-sm font-medium" id="step-indicator-3">3</div>
                                    <span class="text-sm font-medium text-gray-400">Registration</span>
                                </div>
                            </div>
                        </div>

                        <!-- Step 1: Phone Number Check -->
                        <div class="step active" id="step-1">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-blue-500 to-indigo-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-mobile-alt text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Mobile Number Verification</h3>
                                <p class="text-gray-600">Enter your mobile number to check existing registration</p>
                            </div>
                            
                            <div class="max-w-md mx-auto">
                                <div class="mb-6">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Mobile Number</label>
                                    <input type="tel" id="check_phone" placeholder="Enter your mobile number" required
                                           class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                </div>
                                
                                <button type="button" onclick="checkExistingDoctor()" 
                                        class="w-full px-6 py-3 bg-gradient-to-r from-blue-500 to-indigo-600 text-white font-semibold rounded-xl hover:from-blue-600 hover:to-indigo-700 transform hover:scale-105 transition-all duration-300 shadow-lg">
                                    <i class="fas fa-search mr-2"></i>Check Existing Doctor
                                </button>
                                
                                <!-- Phone validation result -->
                                <div id="phoneCheckResult" style="display: none;" class="mt-6 p-4 rounded-xl border">
                                    <!-- Results will be shown here -->
                                </div>
                            </div>
                        </div>

                        <!-- Step 2: Hospital Selection -->
                        <div class="step" id="step-2">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-indigo-500 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-hospital text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Select Hospital</h3>
                                <p class="text-gray-600">Choose the hospital you want to join</p>
                            </div>
                            
                            <form class="space-y-6">
                                <div class="grid md:grid-cols-2 gap-6">
                                    <?php 
                                    // Debug output
                                    echo "<!-- Debug: Hospitals count: " . (isset($hospitals) ? count($hospitals) : 0) . " -->";
                                    if (isset($hospitals) && !empty($hospitals)): 
                                        foreach ($hospitals as $hospital): ?>
                                        <div class="hospital-card border-2 border-gray-200 rounded-xl p-6 hover:border-primary transition-all duration-300" 
                                             data-hospital-id="<?php echo $hospital->id; ?>" 
                                             data-hospital-name="<?php echo htmlspecialchars($hospital->name); ?>">
                                            <div class="text-center">
                                                <div class="w-16 h-16 bg-gradient-to-r from-blue-500 to-indigo-600 rounded-full flex items-center justify-center mx-auto mb-4">
                                                    <i class="fas fa-hospital text-2xl text-white"></i>
                                                </div>
                                                <h4 class="text-lg font-semibold text-gray-800 mb-2"><?php echo htmlspecialchars($hospital->name); ?></h4>
                                                <p class="text-sm text-gray-600 mb-4"><?php echo htmlspecialchars($hospital->address); ?></p>
                                                <div class="text-xs text-gray-500">
                                                    <i class="fas fa-phone mr-1"></i><?php echo htmlspecialchars($hospital->phone); ?>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; 
                                    else: ?>
                                        <div class="text-center py-8 col-span-2">
                                            <i class="fas fa-hospital text-gray-400 text-4xl mb-4"></i>
                                            <p class="text-gray-500">No hospitals available for registration.</p>
                                            <p class="text-sm text-gray-400">Please contact support for assistance.</p>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <?php if (isset($hospitals)): ?>
                                    <p class="text-sm text-blue-600 text-center">Available hospitals: <?php echo count($hospitals); ?></p>
                                <?php else: ?>
                                    <p class="text-sm text-red-600 text-center">No hospitals variable found</p>
                                <?php endif; ?>
                                
                                <div class="flex justify-between">
                                    <button type="button" onclick="goToStep(1)" 
                                            class="px-8 py-3 border-2 border-gray-300 text-gray-700 font-semibold rounded-xl hover:border-primary hover:text-primary transition-all duration-300">
                                        <i class="fas fa-arrow-left mr-2"></i>Back
                                    </button>
                                    <button type="button" id="continueToRegistration" disabled
                                            class="px-8 py-3 bg-gradient-to-r from-indigo-500 to-blue-600 text-white font-semibold rounded-xl hover:from-indigo-600 hover:to-blue-700 transform hover:scale-105 transition-all duration-300 shadow-lg disabled:opacity-50 disabled:cursor-not-allowed">
                                        <i class="fas fa-arrow-right mr-2"></i>Continue to Registration
                                    </button>
                                </div>
                            </form>
                        </div>

                        <!-- Step 3: Registration Form -->
                        <div class="step" id="step-3">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-purple-500 to-pink-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-user-md text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Complete Registration</h3>
                                <div id="selected-hospital-info" class="bg-blue-50 border border-blue-200 rounded-xl p-4 max-w-md mx-auto mb-4">
                                    <!-- Selected hospital info will be shown here -->
                                </div>
                                <p class="text-gray-600">Please fill in your details to complete registration</p>
                            </div>
                            
                            <form id="registrationForm" class="space-y-6">
                                <input type="hidden" id="selected_hospital_id" name="hospital_id" value="">
                                <input type="hidden" id="registration_phone" name="phone" value="">
                                
                                <div class="grid md:grid-cols-2 gap-6">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Full Name <span class="text-red-500">*</span></label>
                                        <input type="text" name="name" required placeholder="Enter your full name"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Phone Number <span class="text-red-500">*</span></label>
                                        <input type="tel" name="phone" id="registration_phone_visible" required placeholder="Enter your phone number"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                        <div id="phone_validation_message" class="mt-2 text-sm" style="display: none;"></div>
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                                        <input type="email" name="email" placeholder="Enter your email (optional)"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Speciality</label>
                                        <select name="department" id="speciality_select" disabled
                                                class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300 bg-gray-100">
                                            <option value="">Please select a hospital first</option>
                                        </select>
                                        <div id="speciality_loading" class="mt-2 text-sm text-blue-600" style="display: none;">
                                            <i class="fas fa-spinner fa-spin mr-1"></i>Loading specialities...
                                        </div>
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Qualification <span class="text-red-500">*</span></label>
                                        <input type="text" name="profile" required placeholder="Enter your qualifications"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300">
                                    </div>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Address <span class="text-red-500">*</span></label>
                                    <textarea name="address" required placeholder="Enter your complete address" rows="3"
                                              class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300"></textarea>
                                </div>
                                
                                <div class="flex flex-col sm:flex-row gap-4 justify-between pt-6">
                                    <button type="button" onclick="goToStep(2)" 
                                            class="px-8 py-3 border-2 border-gray-300 text-gray-700 font-semibold rounded-xl hover:border-primary hover:text-primary transition-all duration-300">
                                        <i class="fas fa-arrow-left mr-2"></i>Back to Hospital Selection
                                    </button>
                                    <button type="submit" 
                                            class="px-8 py-3 bg-gradient-to-r from-purple-500 to-pink-600 text-white font-semibold rounded-xl hover:from-purple-600 hover:to-pink-700 transform hover:scale-105 transition-all duration-300 shadow-lg">
                                        <i class="fas fa-user-md mr-2"></i>Complete Registration
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <script>
        function goToStep(stepNumber) {
            // Hide all steps
            $('.step').removeClass('active');
            
            // Reset all indicators
            for (let i = 1; i <= 3; i++) { // We have 3 steps
                const indicator = document.getElementById(`step-indicator-${i}`);
                
                if (i < stepNumber) {
                    indicator.className = 'step-circle w-8 h-8 bg-green-500 text-white rounded-full flex items-center justify-center text-sm font-medium';
                } else if (i === stepNumber) {
                    indicator.className = 'step-circle w-8 h-8 bg-primary text-white rounded-full flex items-center justify-center text-sm font-medium pulse-animation';
                } else {
                    indicator.className = 'step-circle w-8 h-8 bg-gray-200 text-gray-400 rounded-full flex items-center justify-center text-sm font-medium';
                }
            }
            
            // Update progress bars (we have 2 progress bars between 3 steps)
            for (let i = 1; i <= 2; i++) {
                const progressBar = document.getElementById(`progress-${i}`);
                if (progressBar) {
                    if (i < stepNumber - 1) {
                        progressBar.style.width = '100%'; // Full progress
                    } else {
                        progressBar.style.width = '0%'; // No progress
                    }
                }
            }
            
            // Show target step
            $(`#step-${stepNumber}`).addClass('active');
        }

        function resetForm() {
            $('#check_phone').val('');
            $('#registration_phone').val('');
            $('#registration_phone_visible').val('');
            $('#selected_hospital_id').val('');
            $('#selected-hospital-info').html('');
            $('.hospital-card').removeClass('selected');
            $('#continueToRegistration').prop('disabled', true);
            $('#phone_validation_message').hide();
            // Reset speciality dropdown
            $('#speciality_select').prop('disabled', true).addClass('bg-gray-100');
            $('#speciality_select').html('<option value="">Please select a hospital first</option>');
            $('#speciality_loading').hide();
            goToStep(1);
        }

        let phoneVerified = false;
        let verifiedPhone = '';
        let isPhoneValid = false;

        function checkExistingDoctor() {
            const phone = $('#check_phone').val().trim();
            
            if (!phone) {
                Swal.fire('Error', 'Please enter a phone number', 'error');
                return;
            }

            // Hide any previous results
            $('#phoneCheckResult').hide();

            $.ajax({
                url: '<?php echo base_url('frontend/check_existing_doctor_by_phone'); ?>',
                method: 'POST',
                data: { phone: phone },
                dataType: 'json',
                success: function(response) {
                    if (response.exists) {
                        if (response.is_doctor) {
                            // User exists and is a doctor
                            const doctor = response.doctor;
                            $('#phoneCheckResult').html(`
                                <div class="bg-green-50 border-green-200">
                                    <div class="text-center">
                                        <i class="fas fa-user-md text-3xl text-green-600 mb-3"></i>
                                        <h4 class="text-xl font-semibold text-green-800 mb-3">Doctor Found!</h4>
                                        <div class="space-y-2 text-green-700 mb-4">
                                            <p><strong>Name:</strong> ${doctor.name}</p>
                                            <p><strong>Phone:</strong> ${doctor.phone}</p>
                                            <p><strong>Speciality:</strong> ${doctor.department}</p>
                                            <p><strong>Hospital:</strong> ${doctor.hospital_name}</p>
                                        </div>
                                        <div class="space-y-2">
                                            <button onclick="window.location.href='<?php echo base_url('auth/login'); ?>'" 
                                                    class="w-full px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                                                <i class="fas fa-sign-in-alt mr-2"></i>Go to Login
                                            </button>
                                            <button onclick="resetForm()" 
                                                    class="w-full px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400">
                                                Try Different Number
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            `).show();
                        } else {
                            // User exists but is not a doctor (patient, admin, etc.)
                            $('#phoneCheckResult').html(`
                                <div class="bg-red-50 border-red-200">
                                    <div class="text-center">
                                        <i class="fas fa-exclamation-triangle text-3xl text-red-600 mb-3"></i>
                                        <h4 class="text-xl font-semibold text-red-800 mb-3">Phone Number Already in Use</h4>
                                        <p class="text-red-700 mb-4">${response.message}</p>
                                        <button onclick="resetForm()" 
                                                class="w-full px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                                            Try Different Number
                                        </button>
                                    </div>
                                </div>
                            `).show();
                        }
                    } else {
                        // Phone number is available - show verification option
                        $('#phoneCheckResult').html(`
                            <div class="bg-blue-50 border-blue-200">
                                <div class="text-center">
                                    <i class="fas fa-check-circle text-3xl text-blue-600 mb-3"></i>
                                    <h4 class="text-xl font-semibold text-blue-800 mb-3">Phone Number Available!</h4>
                                    <p class="text-blue-700 mb-4">This phone number is available for registration. Please verify ownership to continue.</p>
                                    <button onclick="startPhoneVerification('${phone}')" 
                                            class="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                                        <i class="fas fa-shield-alt mr-2"></i>Verify Phone Number
                                    </button>
                                </div>
                            </div>
                        `).show();
                    }
                },
                error: function() {
                    $('#phoneCheckResult').html(`
                        <div class="bg-red-50 border-red-200">
                            <div class="text-center">
                                <i class="fas fa-exclamation-triangle text-3xl text-red-600 mb-3"></i>
                                <h4 class="text-xl font-semibold text-red-800 mb-3">Error</h4>
                                <p class="text-red-700 mb-4">Something went wrong. Please try again.</p>
                                <button onclick="$('#phoneCheckResult').hide()" 
                                        class="w-full px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                                    Try Again
                                </button>
                            </div>
                        </div>
                    `).show();
                }
            });
        }

        function startPhoneVerification(phone) {
            // Show loading
            // Don't send OTP immediately - just show the two-step modal
            console.log('📱 Opening doctor OTP verification modal for phone check...');
            showDoctorOtpModal(phone, 'phone_check');
        }

        function onPhoneVerified(phone) {
            phoneVerified = true;
            verifiedPhone = phone;
            isPhoneValid = true; // ✅ IMPORTANT: Mark phone as valid after OTP verification
            
            // Update the result to show verified status
            $('#phoneCheckResult').html(`
                <div class="bg-green-50 border-green-200">
                    <div class="text-center">
                        <i class="fas fa-check-circle text-3xl text-green-600 mb-3"></i>
                        <h4 class="text-xl font-semibold text-green-800 mb-3">Phone Verified Successfully!</h4>
                        <p class="text-green-700 mb-4">Your phone number ${phone} has been verified. You can now proceed to register.</p>
                        <button onclick="proceedToHospitalSelection()" 
                                class="w-full px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                            <i class="fas fa-arrow-right mr-2"></i>Continue to Doctor Registration
                        </button>
                    </div>
                </div>
            `);
        }

        function proceedToHospitalSelection() {
            if (!phoneVerified) {
                Swal.fire('Error', 'Please verify your phone number first', 'error');
                return;
            }
            
            // Set the verified phone in the registration form and lock it
            $('#registration_phone').val(verifiedPhone);
            $('#registration_phone_visible').val(verifiedPhone)
                .prop('readonly', true)
                .prop('disabled', true)
                .addClass('bg-gray-100 cursor-not-allowed border-green-500');
            
            // Add verified badge if not already present
            if (!$('#registration_phone_visible').next('.verified-badge').length) {
                $('#registration_phone_visible').after(
                    '<div class="verified-badge flex items-center mt-2 text-green-600 font-semibold">' +
                    '<i class="fas fa-shield-check mr-2"></i>' +
                    '<span>Phone number verified and locked</span>' +
                    '</div>'
                );
            }
            
            // Since phone is verified, mark it as valid for form submission
            isPhoneValid = true;
            
            // Disable phone number input handler after verification to prevent changes
            $('#registration_phone_visible').off('input keyup blur');
            
            // Go to hospital selection step
            goToStep(2);
        }

        function resetForm() {
            $('#check_phone').val('');
            $('#phoneCheckResult').hide();
            phoneVerified = false;
            verifiedPhone = '';
            isPhoneValid = false;
        }

        $(document).ready(function() {
            // Hospital card selection
            $('.hospital-card').on('click', function() {
                $('.hospital-card').removeClass('selected');
                $(this).addClass('selected');
                
                const hospitalId = $(this).data('hospital-id');
                const hospitalName = $(this).data('hospital-name');
                
                $('#selected_hospital_id').val(hospitalId);
                $('#selected-hospital-info').html(`
                    <div class="text-center">
                        <h4 class="font-semibold text-gray-800">${hospitalName}</h4>
                        <p class="text-sm text-gray-600">Selected Hospital</p>
                    </div>
                `);
                
                // Load specialities for selected hospital
                loadSpecialitiesByHospital(hospitalId);
                
                $('#continueToRegistration').prop('disabled', false);
            });

            // Function to load specialities by hospital
            function loadSpecialitiesByHospital(hospitalId) {
                const specialitySelect = $('#speciality_select');
                const loadingDiv = $('#speciality_loading');
                
                // Show loading and disable select
                loadingDiv.show();
                specialitySelect.prop('disabled', true).addClass('bg-gray-100');
                specialitySelect.html('<option value="">Loading specialities...</option>');
                
                $.ajax({
                    url: '<?php echo base_url('frontend/get_specialities_by_hospital'); ?>',
                    method: 'POST',
                    data: { hospital_id: hospitalId },
                    dataType: 'json',
                    success: function(response) {
                        loadingDiv.hide();
                        specialitySelect.prop('disabled', false).removeClass('bg-gray-100');
                        
                        if (response && response.length > 0) {
                            let options = '<option value="">Select your speciality</option>';
                            response.forEach(function(speciality) {
                                options += `<option value="${speciality.name}">${speciality.name}</option>`;
                            });
                            specialitySelect.html(options);
                        } else {
                            specialitySelect.html('<option value="">No specialities available for this hospital</option>');
                        }
                    },
                    error: function() {
                        loadingDiv.hide();
                        specialitySelect.prop('disabled', false).removeClass('bg-gray-100');
                        specialitySelect.html('<option value="">Error loading specialities</option>');
                        Swal.fire('Error', 'Failed to load specialities. Please try again.', 'error');
                    }
                });
            }

            // Continue to registration button click
            $('#continueToRegistration').on('click', function() {
                const phone = $('#check_phone').val();
                const hospitalId = $('#selected_hospital_id').val();
                
                $('#registration_phone').val(phone);
                $('#registration_phone_visible').val(phone);
                
                goToStep(3);
                
                // Load specialities again for the registration step
                if (hospitalId) {
                    loadSpecialitiesByHospital(hospitalId);
                }
                
                // Trigger phone validation for the pre-filled phone number
                setTimeout(function() {
                    $('#registration_phone_visible').trigger('input');
                }, 500);
            });

            // Phone number validation
            let phoneCheckTimeout;
            
            $('#registration_phone_visible').on('input', function() {
                const phone = $(this).val().trim();
                const messageDiv = $('#phone_validation_message');
                const submitButton = $('#registrationForm button[type="submit"]');
                
                // Clear previous timeout
                clearTimeout(phoneCheckTimeout);
                
                if (phone.length < 10) {
                    messageDiv.hide();
                    isPhoneValid = false;
                    return;
                }
                
                // Show checking message
                messageDiv.html('<i class="fas fa-spinner fa-spin text-blue-500"></i> <span class="text-blue-500">Checking phone number...</span>').show();
                
                // Delay the check to avoid too many requests
                phoneCheckTimeout = setTimeout(function() {
                    $.ajax({
                        url: '<?php echo base_url('frontend/check_phone_exists'); ?>',
                        method: 'POST',
                        data: { phone: phone },
                        dataType: 'json',
                        success: function(response) {
                            if (response.exists) {
                                messageDiv.html('<i class="fas fa-times-circle text-red-500"></i> <span class="text-red-500">' + response.message + '</span>').show();
                                submitButton.prop('disabled', true).addClass('opacity-50 cursor-not-allowed');
                                isPhoneValid = false;
                            } else {
                                messageDiv.html('<i class="fas fa-check-circle text-green-500"></i> <span class="text-green-500">Phone number is available</span>').show();
                                submitButton.prop('disabled', false).removeClass('opacity-50 cursor-not-allowed');
                                isPhoneValid = true;
                            }
                        },
                        error: function() {
                            messageDiv.html('<i class="fas fa-exclamation-triangle text-orange-500"></i> <span class="text-orange-500">Could not verify phone number</span>').show();
                            isPhoneValid = false;
                        }
                    });
                }, 800); // Wait 800ms after user stops typing
            });

            // OLD REGISTRATION HANDLER - REMOVED TO PREVENT CONFLICTS
            // This has been replaced by the OTP-based registration handler below

            // Mobile menu functionality
            document.getElementById('mobile-menu-button').addEventListener('click', function() {
                const mobileMenu = document.getElementById('mobile-menu');
                const icon = this.querySelector('i');
                
                if (mobileMenu.classList.contains('hidden')) {
                    mobileMenu.classList.remove('hidden');
                    icon.classList.remove('fa-bars');
                    icon.classList.add('fa-times');
                } else {
                    mobileMenu.classList.add('hidden');
                    icon.classList.remove('fa-times');
                    icon.classList.add('fa-bars');
                }
            });
        });
    </script>

    <!-- Doctor Registration OTP Modal (Pure Tailwind) -->
    <div id="doctorOtpModal" class="fixed inset-0 z-50 hidden" aria-labelledby="doctorOtpModalLabel" role="dialog" aria-modal="true">
        <!-- Background overlay -->
        <div class="fixed inset-0 bg-gray-500 bg-opacity-75 modal-overlay" onclick="closeDoctorOtpModal()"></div>
        
        <!-- Modal container - perfectly centered -->
        <div class="fixed inset-0 flex items-center justify-center p-4">
            <!-- Modal panel -->
            <div class="bg-white rounded-lg shadow-xl max-w-lg w-full max-h-screen overflow-y-auto">
                <!-- Modal Header -->
                <div class="bg-gradient-to-r from-blue-600 to-purple-600 px-6 py-4">
                    <div class="flex items-center justify-between">
                        <h3 class="text-lg font-medium text-white" id="doctorOtpModalLabel">
                            <i class="fas fa-user-md mr-2"></i>Doctor Phone Verification
                        </h3>
                        <button type="button" class="text-white hover:text-gray-200 focus:outline-none" onclick="closeDoctorOtpModal()">
                            <span class="sr-only">Close</span>
                            <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                            </svg>
                        </button>
                    </div>
                </div>
                
                <!-- Modal Body -->
                <div class="px-6 py-6">
                    <!-- Step 1: Send OTP -->
                    <div id="otp_step_1" class="text-center">
                        <div class="w-20 h-20 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-mobile-alt text-3xl text-white"></i>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800 mb-3">Verify Your Phone Number</h4>
                        <p class="text-gray-600 mb-3">We need to verify your phone number to continue with the registration.</p>
                        <p class="text-lg font-semibold text-blue-600 mb-6" id="doctor_phone_display"></p>
                        
                        <button type="button" class="bg-blue-600 hover:bg-blue-700 text-white font-medium py-3 px-8 rounded-lg transition duration-200" id="send_doctor_otp_btn">
                            <i class="fas fa-paper-plane mr-2"></i>Send Verification Code
                        </button>
                    </div>

                    <!-- Step 2: Enter OTP -->
                    <div id="otp_step_2" class="text-center hidden">
                        <div class="w-20 h-20 bg-gradient-to-r from-green-500 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-sms text-3xl text-white"></i>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800 mb-2">Verification Code Sent</h4>
                        <p class="text-gray-600 mb-1">Enter the 6-digit code sent to</p>
                        <p class="text-lg font-semibold text-blue-600 mb-6" id="doctor_otp_phone_display"></p>

                        <!-- OTP Input -->
                        <form id="doctorOtpVerificationForm">
                            <div class="flex justify-center gap-2 mb-4">
                                <input type="text" class="doctor-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-blue-500 focus:outline-none transition duration-200" maxlength="1" id="doctor_otp1" data-index="1" tabindex="1">
                                <input type="text" class="doctor-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-blue-500 focus:outline-none transition duration-200" maxlength="1" id="doctor_otp2" data-index="2" tabindex="2">
                                <input type="text" class="doctor-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-blue-500 focus:outline-none transition duration-200" maxlength="1" id="doctor_otp3" data-index="3" tabindex="3">
                                <input type="text" class="doctor-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-blue-500 focus:outline-none transition duration-200" maxlength="1" id="doctor_otp4" data-index="4" tabindex="4">
                                <input type="text" class="doctor-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-blue-500 focus:outline-none transition duration-200" maxlength="1" id="doctor_otp5" data-index="5" tabindex="5">
                                <input type="text" class="doctor-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-blue-500 focus:outline-none transition duration-200" maxlength="1" id="doctor_otp6" data-index="6" tabindex="6">
                            </div>
                            <p class="text-sm text-gray-500 mb-4">Enter the 6-digit verification code</p>

                            <!-- Timer and Resend -->
                            <div class="mb-4">
                                <div id="doctor_otp_timer_container" class="hidden">
                                    <p class="text-gray-600 mb-2">
                                        <i class="fas fa-clock mr-1"></i>
                                        Code expires in: <span id="doctor_otp_timer" class="font-semibold text-yellow-600">05:00</span>
                                    </p>
                                </div>
                                <div id="doctor_otp_resend_container" class="hidden">
                                    <p class="text-gray-600 mb-2">Didn't receive the code?</p>
                                    <button type="button" class="text-blue-600 hover:text-blue-800 font-semibold focus:outline-none" id="doctor_resend_otp_btn">
                                        <i class="fas fa-redo mr-1"></i>Resend Code
                                    </button>
                                </div>
                            </div>

                            <!-- Message Area -->
                            <div id="doctor_otp_message" class="hidden p-3 rounded-lg mb-4"></div>

                            <!-- Action Buttons -->
                            <div class="flex justify-center gap-4">
                                <button type="button" class="bg-gray-500 hover:bg-gray-600 text-white font-medium py-2 px-6 rounded-lg transition duration-200" onclick="showOtpStep1()">
                                    <i class="fas fa-arrow-left mr-2"></i>Back
                                </button>
                                <button type="submit" class="bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-6 rounded-lg transition duration-200" id="verify_doctor_otp_btn">
                                    <i class="fas fa-check mr-2"></i>Verify Code
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Include Original OTP Verification Modal (for other purposes if needed) -->
    <?php $this->load->view('includes/otp_verification_modal'); ?>
    
    <script>
        // Doctor OTP Modal Variables
        let currentDoctorPhone = '';
        let currentDoctorPurpose = '';
        let doctorOtpTimer = null;
        let doctorOtpTimeLeft = 300; // 5 minutes

        // Show Doctor OTP Modal (Step 1)
        function showDoctorOtpModal(phone, purpose) {
            currentDoctorPhone = phone;
            currentDoctorPurpose = purpose;
            
            $('#doctor_phone_display').text(phone);
            $('#doctor_otp_phone_display').text(phone);
            
            // Show step 1, hide step 2
            showOtpStep1();
            
            // Show modal with Tailwind classes
            $('#doctorOtpModal').removeClass('hidden');
            $('body').addClass('overflow-hidden'); // Prevent background scrolling
        }

        // Close Doctor OTP Modal
        function closeDoctorOtpModal() {
            $('#doctorOtpModal').addClass('hidden');
            $('body').removeClass('overflow-hidden'); // Restore background scrolling
            
            // Clear timer if running
            if (doctorOtpTimer) {
                clearInterval(doctorOtpTimer);
                doctorOtpTimer = null;
            }
            
            // Reset form
            resetDoctorOtpForm();
        }

        // Show Step 1 (Send OTP button)
        function showOtpStep1() {
            $('#otp_step_1').removeClass('hidden');
            $('#otp_step_2').addClass('hidden');
            $('#doctorOtpModalLabel').html('<i class="fas fa-user-md mr-2"></i>Doctor Phone Verification');
        }

        // Show Step 2 (OTP input fields)
        function showOtpStep2() {
            $('#otp_step_1').addClass('hidden');
            $('#otp_step_2').removeClass('hidden');
            $('#doctorOtpModalLabel').html('<i class="fas fa-sms mr-2"></i>Enter Verification Code');
            
            // Focus on first OTP input and debug
            setTimeout(() => {
                console.log('🔍 Checking OTP inputs existence:');
                for (let i = 1; i <= 6; i++) {
                    let input = $('#doctor_otp' + i);
                    console.log(`doctor_otp${i}:`, input.length > 0 ? 'EXISTS' : 'NOT FOUND');
                }
                $('#doctor_otp1').focus();
            }, 100);
            
            // Start timer
            startDoctorOtpTimer();
        }

        // Reset OTP form
        function resetDoctorOtpForm() {
            // Clear OTP inputs
            for (let i = 1; i <= 6; i++) {
                $('#doctor_otp' + i).val('');
            }
            
            // Hide message
            $('#doctor_otp_message').addClass('hidden');
            
            // Reset timer containers
            $('#doctor_otp_timer_container').addClass('hidden');
            $('#doctor_otp_resend_container').addClass('hidden');
            
            // Reset button states
            $('#send_doctor_otp_btn').prop('disabled', false).html('<i class="fas fa-paper-plane mr-2"></i>Send Verification Code');
            $('#verify_doctor_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-2"></i>Verify Code');
        }

        // Send OTP (triggered by button click in step 1)
        function sendDoctorOtp() {
            $('#send_doctor_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Sending...');
            
            $.ajax({
                url: '<?php echo base_url('frontend/send_otp'); ?>',
                method: 'POST',
                data: {
                    phone: currentDoctorPhone,
                    purpose: 'doctor_registration',
                    hospital_id: $('#selected_hospital_id').val() || ''
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        console.log('✅ OTP sent successfully');
                        showOtpStep2();
                    } else {
                        Swal.fire('Error', response.message || 'Failed to send verification code', 'error');
                    }
                },
                error: function() {
                    Swal.fire('Error', 'Network error. Please try again.', 'error');
                },
                complete: function() {
                    $('#send_doctor_otp_btn').prop('disabled', false).html('<i class="fas fa-paper-plane mr-2"></i>Send Verification Code');
                }
            });
        }

        // Timer functions
        function startDoctorOtpTimer() {
            doctorOtpTimeLeft = 300; // 5 minutes
            $('#doctor_otp_timer_container').removeClass('hidden');
            $('#doctor_otp_resend_container').addClass('hidden');
            
            doctorOtpTimer = setInterval(function() {
                doctorOtpTimeLeft--;
                updateDoctorOtpTimer();
                
                if (doctorOtpTimeLeft <= 0) {
                    clearInterval(doctorOtpTimer);
                    $('#doctor_otp_timer_container').addClass('hidden');
                    $('#doctor_otp_resend_container').removeClass('hidden');
                }
            }, 1000);
        }

        function updateDoctorOtpTimer() {
            let minutes = Math.floor(doctorOtpTimeLeft / 60);
            let seconds = doctorOtpTimeLeft % 60;
            $('#doctor_otp_timer').text(
                (minutes < 10 ? '0' : '') + minutes + ':' + 
                (seconds < 10 ? '0' : '') + seconds
            );
        }

        // OTP input handling
        function setupDoctorOtpInputs() {
            // Handle input navigation for doctor OTP inputs using class
            $('.doctor-otp-input').on('input', function() {
                if (this.value.length === 1) {
                    let currentIndex = parseInt($(this).data('index'));
                    if (currentIndex < 6) {
                        $('#doctor_otp' + (currentIndex + 1)).focus();
                    }
                }
            });

            $('.doctor-otp-input').on('keydown', function(e) {
                if (e.key === 'Backspace' && this.value.length === 0) {
                    let currentIndex = parseInt($(this).data('index'));
                    if (currentIndex > 1) {
                        $('#doctor_otp' + (currentIndex - 1)).focus();
                    }
                }
            });

            $('.doctor-otp-input').on('paste', function(e) {
                e.preventDefault();
                let paste = (e.originalEvent.clipboardData || window.clipboardData).getData('text');
                if (paste.length === 6 && /^\d+$/.test(paste)) {
                    $('.doctor-otp-input').each(function(index) {
                        $(this).val(paste.charAt(index));
                    });
                    $('#doctor_otp6').focus();
                }
            });
        }

        // Initialize when document is ready
        $(document).ready(function() {
            console.log('🔧 Doctor OTP Modal initialized');
            
            setupDoctorOtpInputs();
            
            // Send OTP button click
            $('#send_doctor_otp_btn').click(function() {
                sendDoctorOtp();
            });
            
            // Resend OTP button click
            $('#doctor_resend_otp_btn').click(function() {
                sendDoctorOtp();
            });
            
            // OTP form submission
            $('#doctorOtpVerificationForm').submit(function(e) {
                e.preventDefault();
                verifyDoctorOtp();
            });
            
            // Keyboard event handling
            $(document).keydown(function(e) {
                if (e.key === 'Escape' && !$('#doctorOtpModal').hasClass('hidden')) {
                    closeDoctorOtpModal();
                }
            });
        });

        // Verify OTP
        function verifyDoctorOtp() {
            let otp = '';
            console.log('🔍 Collecting OTP from inputs...');
            
            // Collect OTP from inputs using class selector
            $('.doctor-otp-input').each(function() {
                let value = $(this).val() || '';
                console.log(`Input ${$(this).attr('id')}: "${value}"`);
                otp += value;
            });
            
            console.log('Complete OTP:', otp, 'Length:', otp.length);
            
            if (otp.length !== 6 || !/^\d{6}$/.test(otp)) {
                console.log('❌ OTP validation failed - Length:', otp.length, 'Value:', otp);
                showDoctorOtpMessage('Please enter the complete 6-digit code', 'danger');
                return;
            }
            
            console.log('✅ OTP validation passed, sending to server with otp_code:', otp);
            
            $('#verify_doctor_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Verifying...');
            
            $.ajax({
                url: '<?php echo base_url('frontend/verify_otp'); ?>',
                method: 'POST',
                data: {
                    phone: currentDoctorPhone,
                    otp_code: otp,
                    purpose: 'doctor_registration'
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        showDoctorOtpMessage('✅ Phone verified successfully!', 'success');
                        setTimeout(() => {
                            closeDoctorOtpModal();
                            handleDoctorOtpSuccess();
                        }, 1500);
                    } else {
                        showDoctorOtpMessage(response.message || 'Invalid verification code', 'danger');
                    }
                },
                error: function() {
                    showDoctorOtpMessage('Network error. Please try again.', 'danger');
                },
                complete: function() {
                    $('#verify_doctor_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-2"></i>Verify Code');
                }
            });
        }

        // Handle successful OTP verification
        function handleDoctorOtpSuccess() {
            if (currentDoctorPurpose === 'phone_check') {
                onPhoneVerified(currentDoctorPhone);
            } else if (currentDoctorPurpose === 'final_registration') {
                submitDoctorRegistration();
            }
        }

        // Show message in OTP modal
        function showDoctorOtpMessage(message, type) {
            let bgColor = '';
            let textColor = '';
            let borderColor = '';
            
            if (type === 'success') {
                bgColor = 'bg-green-100';
                textColor = 'text-green-800';
                borderColor = 'border-green-400';
            } else if (type === 'danger' || type === 'error') {
                bgColor = 'bg-red-100';
                textColor = 'text-red-800';
                borderColor = 'border-red-400';
            } else if (type === 'warning') {
                bgColor = 'bg-yellow-100';
                textColor = 'text-yellow-800';
                borderColor = 'border-yellow-400';
            } else {
                bgColor = 'bg-blue-100';
                textColor = 'text-blue-800';
                borderColor = 'border-blue-400';
            }
            
            $('#doctor_otp_message')
                .removeClass()
                .addClass(`${bgColor} ${textColor} ${borderColor} border p-3 rounded-lg mb-4`)
                .text(message)
                .removeClass('hidden');
                
            setTimeout(() => {
                $('#doctor_otp_message').addClass('hidden');
            }, 5000);
        }
        
        // OTP verification success callback
        window.otpVerificationSuccess = function(phone, purpose) {
            if (purpose === 'doctor_registration') {
                // Check if this is for initial phone verification or final registration
                if (!phoneVerified) {
                    // Initial phone verification - update UI and allow proceeding
                    onPhoneVerified(phone);
                } else {
                    // Final registration verification - proceed with registration
                    submitDoctorRegistration();
                }
            }
        };

        // Modify the existing registration form submission
        let registrationFormData = null;

        // Override the original form submission
        $('#registrationForm').off('submit').on('submit', function(e) {
            e.preventDefault();
            
            // Store form data
            registrationFormData = $(this).serialize();
            
            // Basic client-side validation (keep existing validation)
            const name = $('input[name="name"]').val().trim();
            const phone = $('#registration_phone_visible').val().trim();
            const department = $('select[name="department"]').val();
            const profile = $('input[name="profile"]').val().trim();
            const address = $('textarea[name="address"]').val().trim();
            
            if (!name) {
                Swal.fire('Validation Error', 'Name is required', 'error');
                return;
            }
            
            if (!phone) {
                Swal.fire('Validation Error', 'Phone number is required', 'error');
                return;
            }
            
            // If phone is verified through OTP, it's automatically valid
            // Only check isPhoneValid if phone was NOT verified through OTP (fallback validation)
            if (!phoneVerified && !isPhoneValid) {
                Swal.fire('Validation Error', 'Please enter a valid and available phone number', 'error');
                return;
            }
            
            if (!profile) {
                Swal.fire('Validation Error', 'Qualification is required', 'error');
                return;
            }
            
            if (!address) {
                Swal.fire('Validation Error', 'Address is required', 'error');
                return;
            }
            
            // Sync the visible phone number with the hidden field
            $('#registration_phone').val(phone);
            registrationFormData = $(this).serialize();
            
            // Check if OTP is already verified for this phone
            $.ajax({
                url: '<?php echo base_url('frontend/check_verified_otp'); ?>',
                method: 'POST',
                data: {
                    phone: phone,
                    purpose: 'doctor_registration',
                    within_minutes: 30
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success && response.verified) {
                        // Already verified, proceed with registration
                        submitDoctorRegistration();
                    } else {
                        // Need OTP verification, send OTP
                        sendOTPForRegistration(phone);
                    }
                },
                error: function() {
                    // If check fails, proceed with OTP verification
                    sendOTPForRegistration(phone);
                }
            });
        });

        function sendOTPForRegistration(phone) {
            // Don't send OTP immediately - just show the two-step modal
            console.log('📱 Opening doctor OTP verification modal for final registration...');
            showDoctorOtpModal(phone, 'final_registration');
        }

        function submitDoctorRegistration() {
            if (!registrationFormData) {
                Swal.fire('Error', 'Form data not found. Please try again.', 'error');
                return;
            }

            // Show loading
            Swal.fire({
                title: 'Creating Account',
                text: 'Please wait while we create your account...',
                allowOutsideClick: false,
                allowEscapeKey: false,
                showConfirmButton: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            $.ajax({
                url: '<?php echo base_url('frontend/register_doctor'); ?>',
                method: 'POST',
                data: registrationFormData,
                dataType: 'json',
                success: function(response) {
                    Swal.close();
                    
                    if (response.success) {
                        // Send SMS in background (don't wait for it)
                        if (response.sms_data) {
                            $.ajax({
                                url: '<?php echo base_url('frontend/send_doctor_registration_sms'); ?>',
                                method: 'POST',
                                data: response.sms_data,
                                dataType: 'json',
                                success: function(smsResponse) {
                                    console.log('Doctor SMS Status:', smsResponse.message);
                                },
                                error: function() {
                                    console.log('Doctor SMS sending failed (non-critical)');
                                }
                            });
                        }
                        
                        // Show beautiful success message
                        Swal.fire({
                            title: 'Doctor Registration Successful!',
                            html: `
                                <div class="text-center">
                                    <i class="fas fa-user-md text-blue-600 text-4xl mb-4"></i>
                                    <p class="text-lg mb-2">Welcome to our Hospital Management System!</p>
                                    <p class="text-gray-600 mb-4">Your doctor registration has been submitted successfully.</p>
                                    <div class="bg-blue-50 p-4 rounded-lg mb-4">
                                        <p class="font-semibold text-blue-800">Your Login Details:</p>
                                        <p class="text-blue-700">Phone: <strong>${response.sms_data ? response.sms_data.phone : 'N/A'}</strong></p>
                                        <p class="text-blue-700">Password: <strong>${response.sms_data ? response.sms_data.password : 'N/A'}</strong></p>
                                        <p class="text-blue-700">Department: <strong>${response.sms_data ? response.sms_data.department : 'N/A'}</strong></p>
                                        <p class="text-sm text-blue-600 mt-2">📱 SMS with login details has been sent to your phone</p>
                                    </div>
                                    <div class="bg-yellow-50 p-4 rounded-lg mb-4">
                                        <p class="font-semibold text-yellow-800">⏳ Pending Admin Approval</p>
                                        <p class="text-yellow-700 text-sm">Your account will be activated after admin approval. You will be notified once approved.</p>
                                    </div>
                                    <p class="text-gray-600">What would you like to do next?</p>
                                </div>
                            `,
                            icon: 'success',
                            showCancelButton: true,
                            confirmButtonText: '<i class="fas fa-sign-in-alt mr-2"></i>Go to Login',
                            cancelButtonText: '<i class="fas fa-home mr-2"></i>Go to Home',
                            confirmButtonColor: '#2563eb',
                            cancelButtonColor: '#6b7280'
                        }).then((result) => {
                            if (result.isConfirmed) {
                                window.location.href = '<?php echo base_url('auth/login'); ?>';
                            } else {
                                window.location.href = '<?php echo base_url('frontend/public_portal'); ?>';
                            }
                        });
                    } else {
                        Swal.fire('Error', response.message || 'Registration failed. Please try again.', 'error');
                    }
                },
                error: function() {
                    Swal.close();
                    Swal.fire('Error', 'Something went wrong. Please try again.', 'error');
                }
            });
        }
    </script>
</body>
</html> 