<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $settings->title; ?> - Book Appointment</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'primary': '#667eea',
                        'secondary': '#764ba2',
                    }
                }
            }
        }
    </script>
    <style>
        .gradient-bg {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .step {
            display: none;
            animation: fadeIn 0.5s ease-in;
        }
        .step.active {
            display: block;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .selection-card {
            transition: all 0.3s ease;
        }
        .selection-card:hover {
            transform: translateY(-4px);
        }
        .selection-card.selected {
            transform: translateY(-4px);
        }
        .time-slot {
            transition: all 0.3s ease;
        }
        .time-slot:hover {
            transform: translateY(-2px);
        }
        .time-slot.selected {
            transform: translateY(-2px);
        }
        .pulse-animation {
            animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
        }

        /* Custom Flatpickr Styles */
        .flatpickr-calendar {
            border: 1px solid #e5e7eb;
            border-radius: 12px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            font-family: inherit;
        }
        .flatpickr-month {
            background: #667eea;
            color: white;
            border-radius: 12px 12px 0 0;
            padding: 10px;
        }
        .flatpickr-current-month {
            color: white;
        }
        .flatpickr-weekday {
            background: #f8fafc;
            color: #64748b;
            font-weight: 600;
        }
        .flatpickr-day {
            border-radius: 8px;
            transition: all 0.2s ease;
        }
        .flatpickr-day:hover {
            background: #e0e7ff;
            border-color: #667eea;
        }
        .flatpickr-day.selected {
            background: #667eea;
            border-color: #667eea;
        }
        .flatpickr-day.disabled {
            background: #f1f5f9 !important;
            color: #cbd5e1 !important;
            cursor: not-allowed !important;
        }
        .flatpickr-day.available {
            background: #f0fdf4;
            border: 1px solid #16a34a;
            color: #15803d;
            font-weight: 600;
        }
        .flatpickr-day.available:hover {
            background: #dcfce7;
            border-color: #16a34a;
        }
        
        /* Disabled date input styling */
        #appointment_date.no-doctor {
            background-color: #f9fafb;
            color: #9ca3af;
            cursor: not-allowed;
        }
    </style>
</head>
<body class="font-sans bg-gray-50">
    <!-- Navigation -->
    <nav class="backdrop-blur-md shadow-lg fixed w-full top-0 z-50">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-16">
                <!-- Logo -->
                <div class="flex items-center">
                    <a href="<?php echo base_url('frontend/public_portal'); ?>" class="flex items-center">
                        <div class="w-8 h-8 bg-gradient-to-r from-primary to-secondary rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-hospital text-white text-sm"></i>
                        </div>
                        <span class="font-bold text-xl text-white"><?php echo $settings->title; ?></span>
                    </a>
                </div>
                
                <!-- Desktop Navigation -->
                <div class="hidden md:flex items-center space-x-8">
                    <a href="<?php echo base_url('frontend/public_portal'); ?>" class="text-white hover:text-blue-200 transition-colors duration-200 font-medium">
                        <i class="fas fa-home mr-1"></i>Home
                    </a>
                    <a href="<?php echo base_url('frontend/about_us'); ?>" class="text-white hover:text-blue-200 transition-colors duration-200 font-medium">
                        <i class="fas fa-info-circle mr-1"></i>About Us
                    </a>
                    <a href="<?php echo base_url('frontend/our_services'); ?>" class="text-white hover:text-blue-200 transition-colors duration-200 font-medium">
                        <i class="fas fa-medical-kit mr-1"></i>Our Services
                    </a>
                    <a href="<?php echo base_url('auth/login'); ?>" class="bg-gradient-to-r from-primary to-secondary text-white px-6 py-2 rounded-lg hover:from-primary-600 hover:to-secondary-600 transition-all duration-200 font-medium">
                        <i class="fas fa-sign-in-alt mr-1"></i>Login
                    </a>
                </div>
                
                <!-- Mobile Menu Button -->
                <div class="md:hidden">
                    <button id="mobile-menu-button" class="text-white hover:text-blue-200 focus:outline-none">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Mobile Navigation -->
        <div id="mobile-menu" class="hidden md:hidden bg-white border-t border-gray-200">
            <div class="px-2 pt-2 pb-3 space-y-1">
                <a href="<?php echo base_url('frontend/public_portal'); ?>" class="block px-3 py-2 text-gray-700 hover:text-primary hover:bg-gray-50 rounded-md font-medium">
                    <i class="fas fa-home mr-2"></i>Home
                </a>
                <a href="<?php echo base_url('frontend/about_us'); ?>" class="block px-3 py-2 text-gray-700 hover:text-primary hover:bg-gray-50 rounded-md font-medium">
                    <i class="fas fa-info-circle mr-2"></i>About Us
                </a>
                <a href="<?php echo base_url('frontend/our_services'); ?>" class="block px-3 py-2 text-gray-700 hover:text-primary hover:bg-gray-50 rounded-md font-medium">
                    <i class="fas fa-medical-kit mr-2"></i>Our Services
                </a>
                <a href="<?php echo base_url('auth/login'); ?>" class="block px-3 py-2 bg-gradient-to-r from-primary to-secondary text-white rounded-md font-medium">
                    <i class="fas fa-sign-in-alt mr-2"></i>Login
                </a>
            </div>
        </div>
    </nav>
    <section class="gradient-bg min-h-screen py-12 pt-32">
        <div class="container mx-auto px-4">
            <div class="max-w-6xl mx-auto">
                <!-- Header -->
                <div class="text-center text-white mb-8">
                    <h1 class="text-4xl md:text-5xl font-bold mb-4">Book Appointment</h1>
                    <p class="text-xl text-blue-100 mb-6">Schedule your visit with our qualified doctors</p>

                </div>
                
                <!-- Main Card -->
                <div class="bg-white rounded-3xl shadow-2xl overflow-hidden">
                    <div class="p-8 md:p-12">
                        <!-- Step Indicator -->
                        <div class="flex justify-center items-center mb-8 overflow-x-auto">
                            <div class="flex items-center space-x-2 md:space-x-4 min-w-max">
                                <div class="flex items-center">
                                    <div class="w-8 h-8 md:w-10 md:h-10 rounded-full bg-primary text-white flex items-center justify-center font-bold text-sm transition-all duration-300" id="step-indicator-1">1</div>
                                    <div class="ml-1 md:ml-2 text-xs md:text-sm font-medium text-gray-600">Patient</div>
                                </div>
                                <div class="w-8 md:w-12 h-0.5 bg-gray-300" id="line-1"></div>
                                <div class="flex items-center">
                                    <div class="w-8 h-8 md:w-10 md:h-10 rounded-full bg-gray-300 text-gray-600 flex items-center justify-center font-bold text-sm transition-all duration-300" id="step-indicator-2">2</div>
                                    <div class="ml-1 md:ml-2 text-xs md:text-sm font-medium text-gray-600">Doctor Selection</div>
                                </div>
                                <div class="w-8 md:w-12 h-0.5 bg-gray-300" id="line-2"></div>
                                <div class="flex items-center">
                                    <div class="w-8 h-8 md:w-10 md:h-10 rounded-full bg-gray-300 text-gray-600 flex items-center justify-center font-bold text-sm transition-all duration-300" id="step-indicator-3">3</div>
                                    <div class="ml-1 md:ml-2 text-xs md:text-sm font-medium text-gray-600">Date & Time</div>
                                </div>
                                <div class="w-8 md:w-12 h-0.5 bg-gray-300" id="line-3"></div>
                                <div class="flex items-center">
                                    <div class="w-8 h-8 md:w-10 md:h-10 rounded-full bg-gray-300 text-gray-600 flex items-center justify-center font-bold text-sm transition-all duration-300" id="step-indicator-4">4</div>
                                    <div class="ml-1 md:ml-2 text-xs md:text-sm font-medium text-gray-600">Payment</div>
                                </div>
                            </div>
                        </div>

                        <!-- Step 1: Check Patient -->
                        <div class="step active" id="step-1">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-user-search text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Patient Information</h3>
                                <p class="text-gray-600">First, let's verify your patient information</p>
                            </div>
                            
                            <form id="patientCheckForm" class="space-y-6">
                                <div class="max-w-md mx-auto">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">
                                            <i class="fas fa-phone mr-2 text-primary"></i>Phone Number <span class="text-red-500">*</span>
                                        </label>
                                        <input type="tel" id="patient_phone" placeholder="Enter your phone number" required
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300 text-center text-lg"
                                               onkeypress="if(event.key === 'Enter') checkPatientForAppointment()">
                                        <p class="text-sm text-gray-500 mt-2 text-center">
                                            <i class="fas fa-info-circle mr-1"></i>Enter the phone number you used during registration
                                        </p>
                                    </div>
                                </div>
                                <div class="flex flex-col sm:flex-row gap-4 justify-center">
                                    <button type="button" onclick="checkPatientForAppointment()"
                                            class="px-8 py-3 bg-gradient-to-r from-blue-500 to-purple-600 text-white font-semibold rounded-xl hover:from-blue-600 hover:to-purple-700 transform hover:scale-105 transition-all duration-300 shadow-lg">
                                        <i class="fas fa-search mr-2"></i>Find Patient
                                    </button>
                                    <a href="<?php echo base_url('frontend/patient_registration'); ?>"
                                       class="px-8 py-3 border-2 border-gray-300 text-gray-700 font-semibold rounded-xl hover:border-primary hover:text-primary transition-all duration-300 text-center">
                                        <i class="fas fa-user-plus mr-2"></i>Register First
                                    </a>
                                </div>
                            </form>
                            
                            <!-- Phone check result -->
                            <div id="phoneCheckResult" style="display: none;" class="mt-6 p-4 rounded-xl border max-w-md mx-auto">
                                <!-- Results will be shown here -->
                            </div>
                            
                            <div id="patient-info" class="mt-8 hidden">
                                <!-- Patient details will be shown here -->
                            </div>
                        </div>

                        <!-- Step 2: Select Hospital, Specialty & Doctor -->
                        <div class="step" id="step-2">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-green-500 to-teal-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-hospital-user text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Select Hospital, Specialty & Doctor</h3>
                                <p class="text-gray-600">Choose hospital and specialty to filter doctors, or search directly for a doctor</p>
                            </div>
                            
                            <div class="grid lg:grid-cols-3 gap-6 mb-8">
                                <!-- Hospital Selection -->
                                <div class="space-y-4">
                                    <h4 class="text-lg font-semibold text-gray-800 border-b pb-2">
                                        <i class="fas fa-hospital mr-2 text-green-600"></i>Hospital
                                    </h4>
                                    <div class="space-y-2 max-h-60 overflow-y-auto">
                                <?php 
                                // Debug: Check if hospitals data exists
                                if (empty($hospitals)): ?>
                                    <div class="text-center py-8">
                                        <div class="w-12 h-12 bg-yellow-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                            <i class="fas fa-exclamation-triangle text-yellow-600"></i>
                                        </div>
                                        <p class="text-yellow-600 text-sm">No hospitals found in database</p>
                                    </div>
                                <?php else: ?>
                                    <!-- Debug: Show hospital count -->
                                    <script>console.log('Hospitals loaded:', <?php echo count($hospitals); ?>);</script>
                                <?php foreach ($hospitals as $hospital): ?>
                                        <div class="hospital-filter-card bg-white border border-gray-200 rounded-lg p-3 cursor-pointer hover:border-green-500 hover:bg-green-50 transition-all" data-hospital-id="<?php echo $hospital->id; ?>">
                                            <div class="flex items-center">
                                                <div class="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center mr-3">
                                                    <i class="fas fa-hospital text-green-600 text-xs"></i>
                                        </div>
                                                <div>
                                                    <h5 class="font-medium text-gray-800 text-sm"><?php echo $hospital->name; ?></h5>
                                                    <p class="text-xs text-gray-500"><?php echo substr($hospital->address ?? 'Hospital Network', 0, 30); ?></p>
                                                </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>

                                <!-- Specialty Selection -->
                                <div class="space-y-4">
                                    <h4 class="text-lg font-semibold text-gray-800 border-b pb-2">
                                        <i class="fas fa-stethoscope mr-2 text-purple-600"></i>Specialty
                                    </h4>
                                    <div class="space-y-2 max-h-60 overflow-y-auto" id="specialities-filter-container">
                                        <!-- Specialties will be loaded here -->
                                </div>
                            </div>
                            
                                <!-- Doctor Selection with Search -->
                                <div class="space-y-4">
                                    <h4 class="text-lg font-semibold text-gray-800 border-b pb-2">
                                        <i class="fas fa-user-md mr-2 text-blue-600"></i>Doctor
                                    </h4>
                                    
                                    <!-- Doctor Search -->
                                    <div class="relative">
                                        <input type="text" id="doctor-search" placeholder="Search for a doctor..."
                                               class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent pl-10">
                                        <i class="fas fa-search absolute left-3 top-3 text-gray-400"></i>
                            </div>
                            
                                    <!-- Doctor List -->
                                    <div class="space-y-2 max-h-60 overflow-y-auto" id="doctors-filter-container">
                                        <!-- Doctors will be loaded here -->
                                    </div>
                            </div>
                        </div>

                            <!-- Selected Information Display -->
                            <div id="selection-summary" class="bg-gray-50 rounded-lg p-4 mb-6 hidden">
                                <h5 class="font-semibold text-gray-800 mb-2">Your Selection:</h5>
                                <div class="grid md:grid-cols-3 gap-4 text-sm">
                                    <div id="selected-hospital-info" class="hidden">
                                        <span class="text-green-600 font-medium">Hospital:</span>
                                        <span id="selected-hospital-name"></span>
                                </div>
                                    <div id="selected-specialty-info" class="hidden">
                                        <span class="text-purple-600 font-medium">Specialty:</span>
                                        <span id="selected-specialty-name"></span>
                            </div>
                                    <div id="selected-doctor-info" class="hidden">
                                        <span class="text-blue-600 font-medium">Doctor:</span>
                                        <span id="selected-doctor-name"></span>
                        </div>
                                </div>
                            </div>
                            
                            <div class="flex justify-between items-center">
                                <button type="button" onclick="goToStep(1)"
                                        class="px-6 py-3 border-2 border-gray-300 text-gray-700 font-semibold rounded-xl hover:border-gray-400 hover:bg-gray-50 transform hover:scale-105 transition-all duration-300">
                                    <i class="fas fa-arrow-left mr-2"></i>Back to Patient Info
                                </button>
                                <button type="button" id="continueToDateTimeBtn" onclick="continueToDateTime()" disabled
                                        class="px-8 py-3 bg-gradient-to-r from-blue-500 to-purple-600 text-white font-semibold rounded-xl hover:from-blue-600 hover:to-purple-700 transform hover:scale-105 transition-all duration-300 shadow-lg disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none">
                                    <i class="fas fa-arrow-right mr-2"></i>Continue
                                </button>
                            </div>
                        </div>

                        <!-- Step 3: Select Date & Time -->
                        <div class="step" id="step-3">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-orange-500 to-red-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-calendar-alt text-3xl text-white"></i>
                                </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Select Date & Time</h3>
                                <p class="text-gray-600">Choose your preferred appointment date and time slot</p>
                            </div>
                            
                            <!-- Selected Doctor Details -->
                            <div id="selected-doctor-details" class="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-8">
                                <h4 class="text-lg font-semibold text-blue-800 mb-4">
                                    <i class="fas fa-user-md mr-2"></i>Appointment with Selected Doctor
                                </h4>
                                <div class="bg-white rounded-lg p-4">
                                    <div class="flex items-center">
                                        <div class="w-16 h-16 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center mr-4">
                                            <i class="fas fa-user-md text-2xl text-white"></i>
                                        </div>
                                        <div class="flex-1">
                                            <h5 class="text-xl font-semibold text-gray-800 mb-1" id="doctor-detail-name">Doctor Name</h5>
                                            <p class="text-blue-600 font-medium mb-1" id="doctor-detail-specialty">Specialty</p>
                                            <p class="text-gray-600 text-sm" id="doctor-detail-hospital">Hospital</p>
                                            <p class="text-gray-600 text-sm" id="doctor-detail-qualification">Qualification</p>
                                        </div>
                                        <div class="text-right">
                                            <div class="text-lg font-semibold text-green-600" id="doctor-detail-fee">Consultation Fee</div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Visit Type Selection -->
                            <div class="bg-white rounded-xl shadow-lg p-6 mb-8">
                                <h4 class="text-lg font-semibold text-gray-800 mb-6">
                                    <i class="fas fa-clipboard-list mr-2 text-purple-600"></i>Visit Type
                                </h4>
                                <div class="mb-4">
                                    <label class="block text-sm font-medium text-gray-700 mb-3">Select Visit Type *</label>
                                    <select id="visit_type" class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                        <option value="">Select visit type...</option>
                                    </select>
                                </div>
                                <div class="grid md:grid-cols-3 gap-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Visit Charges</label>
                                        <input type="text" id="visit_charges" readonly
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl bg-gray-50 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                               placeholder="$0.00">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Discount</label>
                                        <input type="number" id="discount" value="0" min="0"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                               placeholder="0">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Grand Total</label>
                                        <input type="text" id="grand_total" readonly
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl bg-gray-50 focus:ring-2 focus:ring-blue-500 focus:border-transparent font-semibold text-green-600"
                                               placeholder="$0.00">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="grid md:grid-cols-2 gap-6 mb-8">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Select Date</label>
                                    <div class="relative">
                                        <input type="text" id="appointment_date" readonly placeholder="Select a doctor first to see available dates"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300 cursor-pointer bg-white">
                                        <div class="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                            <i class="fas fa-calendar-alt text-gray-400"></i>
                                        </div>
                                    </div>
                                    <div id="available-dates-info" class="mt-2 text-sm text-gray-600 hidden">
                                        <i class="fas fa-info-circle text-blue-500"></i>
                                        <span id="available-dates-text">Loading available dates...</span>
                                    </div>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Describe Your Problem</label>
                                    <textarea id="appointment_problem" placeholder="Brief description of your health concern" rows="3"
                                              class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-primary focus:border-transparent transition-all duration-300"></textarea>
                                </div>
                            </div>
                            
                            <div class="mb-8">
                                <label class="block text-sm font-medium text-gray-700 mb-4">Available Time Slots</label>
                                <div id="time-slots-container" class="min-h-20">
                                    <div class="text-center text-gray-500 py-8">
                                        <i class="fas fa-clock text-4xl mb-4"></i>
                                        <p>Please select a date to view available time slots</p>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="flex justify-between items-center">
                                <button type="button" onclick="goToStep(2)"
                                        class="px-6 py-3 border-2 border-gray-300 text-gray-700 font-semibold rounded-xl hover:border-gray-400 hover:bg-gray-50 transform hover:scale-105 transition-all duration-300">
                                    <i class="fas fa-arrow-left mr-2"></i>Back to Doctor Selection
                                </button>
                                <button type="button" id="continueToPaymentBtn" onclick="continueToPayment()" disabled
                                        class="px-8 py-3 bg-gradient-to-r from-orange-500 to-red-600 text-white font-semibold rounded-xl hover:from-orange-600 hover:to-red-700 transform hover:scale-105 transition-all duration-300 shadow-lg disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none">
                                    <i class="fas fa-arrow-right mr-2"></i><span id="continue-btn-text">Continue to Payment</span>
                                </button>
                            </div>
                        </div>

                        <!-- Step 4: Payment -->
                        <div class="step" id="step-4">
                            <div class="text-center mb-8">
                                <div class="w-20 h-20 bg-gradient-to-r from-green-500 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <i class="fas fa-credit-card text-3xl text-white"></i>
                    </div>
                                <h3 class="text-2xl font-bold text-gray-800 mb-3">Payment & Confirmation</h3>
                                <p class="text-gray-600">Complete your payment to confirm the appointment</p>
                </div>

                            <!-- Appointment Summary -->
                            <div class="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-8">
                                <h4 class="text-lg font-semibold text-blue-800 mb-4">
                                    <i class="fas fa-clipboard-list mr-2"></i>Appointment Summary
                                </h4>
                                <div class="bg-white rounded-lg p-4">
                                    <div class="grid md:grid-cols-2 gap-4">
                                        <div>
                                            <div class="mb-3">
                                                <span class="font-medium text-gray-600">Patient:</span>
                                                <span class="ml-2 text-gray-800" id="payment-patient-name">Patient Name</span>
                                            </div>
                                            <div class="mb-3">
                                                <span class="font-medium text-gray-600">Doctor:</span>
                                                <span class="ml-2 text-gray-800" id="payment-doctor-name">Doctor Name</span>
                                            </div>
                                            <div class="mb-3">
                                                <span class="font-medium text-gray-600">Specialty:</span>
                                                <span class="ml-2 text-gray-800" id="payment-specialty">Specialty</span>
                                            </div>
                                        </div>
                                        <div>
                                            <div class="mb-3">
                                                <span class="font-medium text-gray-600">Hospital:</span>
                                                <span class="ml-2 text-gray-800" id="payment-hospital">Hospital</span>
                                            </div>
                                            <div class="mb-3">
                                                <span class="font-medium text-gray-600">Date:</span>
                                                <span class="ml-2 text-gray-800" id="payment-date">Date</span>
                                            </div>
                                            <div class="mb-3">
                                                <span class="font-medium text-gray-600">Time:</span>
                                                <span class="ml-2 text-gray-800" id="payment-time">Time</span>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="border-t pt-4 mt-4">
                                        <div class="flex justify-between items-center">
                                            <span class="text-lg font-semibold text-gray-800">Consultation Fee:</span>
                                            <span class="text-2xl font-bold text-green-600" id="payment-amount">$0.00</span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Payment Options -->
                            <div class="bg-white rounded-xl shadow-lg p-6 mb-8">
                                <h4 class="text-lg font-semibold text-gray-800 mb-6">
                                    <i class="fas fa-payment mr-2"></i>Payment Method
                                </h4>
                                
                                <!-- Payment Gateway Selection -->
                                <div class="mb-6">
                                    <label class="block text-sm font-medium text-gray-700 mb-3">Select Payment Method</label>
                                    <div class="grid md:grid-cols-2 gap-4">
                                        <div class="payment-method border-2 border-gray-200 rounded-lg p-4 cursor-pointer hover:border-blue-500 transition-all"
                                             data-method="stripe">
                                            <div class="text-center">
                                                <i class="fab fa-cc-stripe text-3xl text-blue-600 mb-2"></i>
                                                <p class="font-medium text-gray-800">Credit Card</p>
                                                <p class="text-sm text-gray-600">Visa, Mastercard, Amex</p>
                                            </div>
                                        </div>
                                        <div class="payment-method border-2 border-gray-200 rounded-lg p-4 cursor-pointer hover:border-blue-500 transition-all"
                                             data-method="pay_later">
                                            <div class="text-center">
                                                <i class="fas fa-clock text-3xl text-orange-600 mb-2"></i>
                                                <p class="font-medium text-gray-800">Pay Later</p>
                                                <p class="text-sm text-gray-600">Pay at hospital</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Credit Card Form -->
                                <div id="card-payment-form" class="hidden">
                                    <div class="grid md:grid-cols-2 gap-6">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Card Number</label>
                                            <input type="text" id="card_number" placeholder="1234 5678 9012 3456" maxlength="19"
                                                   class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Cardholder Name</label>
                                            <input type="text" id="cardholder_name" placeholder="John Doe"
                                                   class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">Expiry Date</label>
                                            <input type="text" id="expiry_date" placeholder="MM/YY" maxlength="5"
                                                   class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-2">CVV</label>
                                            <input type="text" id="cvv" placeholder="123" maxlength="4"
                                                   class="w-full px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                        </div>
                                    </div>
                                </div>



                                <!-- Pay Later Info -->
                                <div id="pay-later-info" class="hidden">
                                    <div class="bg-orange-50 border border-orange-200 rounded-lg p-4">
                                        <p class="text-orange-800">
                                            <i class="fas fa-info-circle mr-2"></i>
                                            Your appointment will be confirmed and you can pay the consultation fee at the hospital.
                                        </p>
                                    </div>
                                </div>

                                <!-- Optional Payment Info -->
                                <div id="optional-payment-info" class="hidden">
                                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                        <p class="text-blue-800">
                                            <i class="fas fa-info-circle mr-2"></i>
                                            This doctor allows optional payment. You can choose to pay now or at the hospital.
                                        </p>
                                    </div>
                                </div>
                            </div>

                            <div class="flex justify-between items-center">
                                <button type="button" onclick="goToStep(3)"
                                        class="px-6 py-3 border-2 border-gray-300 text-gray-700 font-semibold rounded-xl hover:border-gray-400 hover:bg-gray-50 transform hover:scale-105 transition-all duration-300">
                                    <i class="fas fa-arrow-left mr-2"></i>Back to Date & Time
                                </button>
                                <button type="button" id="processPaymentBtn" onclick="processAppointmentPayment()" disabled
                                        class="px-8 py-3 bg-gradient-to-r from-green-500 to-blue-600 text-white font-semibold rounded-xl hover:from-green-600 hover:to-blue-700 transform hover:scale-105 transition-all duration-300 shadow-lg disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none">
                                    <i class="fas fa-credit-card mr-2"></i>Complete Payment
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <script>
        // Mobile menu functionality
        document.getElementById('mobile-menu-button').addEventListener('click', function() {
            const mobileMenu = document.getElementById('mobile-menu');
            const icon = this.querySelector('i');
            
            if (mobileMenu.classList.contains('hidden')) {
                mobileMenu.classList.remove('hidden');
                icon.classList.remove('fa-bars');
                icon.classList.add('fa-times');
            } else {
                mobileMenu.classList.add('hidden');
                icon.classList.remove('fa-times');
                icon.classList.add('fa-bars');
            }
        });
    </script>
    
    <script>
        let selectedPatient = null;
        let selectedHospital = null;
        let selectedSpeciality = null;
        let selectedDoctor = null;
        let selectedTimeSlot = null;
        let selectedVisitType = null;
        let selectedPaymentMethod = null;
        let doctorPaymentSetting = null;
        let appointmentData = null;
        let allSpecialities = [];
        let allDoctors = [];
        let availableDates = [];
        let datePicker = null;

        function goToStep(stepNumber) {
            $('.step').removeClass('active');
            
            // Update step indicators (always 4 steps now)
            for (let i = 1; i <= 4; i++) {
                const indicator = document.getElementById(`step-indicator-${i}`);
                const line = document.getElementById(`line-${i}`);
                
                if (i < stepNumber) {
                    indicator.className = 'w-8 h-8 md:w-10 md:h-10 rounded-full bg-green-500 text-white flex items-center justify-center font-bold text-sm transition-all duration-300';
                    if (line) line.className = 'w-8 md:w-12 h-0.5 bg-green-500';
                } else if (i === stepNumber) {
                    indicator.className = 'w-8 h-8 md:w-10 md:h-10 rounded-full bg-primary text-white flex items-center justify-center font-bold text-sm transition-all duration-300 pulse-animation';
                    if (line) line.className = 'w-8 md:w-12 h-0.5 bg-gray-300';
                } else {
                    indicator.className = 'w-8 h-8 md:w-10 md:h-10 rounded-full bg-gray-300 text-gray-600 flex items-center justify-center font-bold text-sm transition-all duration-300';
                    if (line) line.className = 'w-8 md:w-12 h-0.5 bg-gray-300';
                }
            }
            
            $(`#step-${stepNumber}`).addClass('active');
            
            // Handle specific step transitions
            if (stepNumber === 2) {
                // Going to doctor selection step - ensure selections are preserved
                restoreSelections();
            } else if (stepNumber === 3) {
                // Going to date/time step - ensure doctor is selected
                if (!selectedDoctor) {
                    Swal.fire('Error', 'Please select a doctor first', 'error');
                    goToStep(2);
                    return;
                }
                // Load doctor details for the date & time step
                loadDoctorDetails();
                // Load visit types for selected doctor
                loadVisitTypes();
            } else if (stepNumber === 4) {
                // Going to payment step - ensure date and time selected
                if (!selectedTimeSlot) {
                    Swal.fire('Error', 'Please select date and time first', 'error');
                    goToStep(3);
                    return;
                }
                // Load appointment summary for payment
                loadAppointmentSummary();
            }
        }
        
        function restoreSelections() {
            // First restore the filtered lists based on current selections
            if (selectedHospital || selectedSpeciality) {
                filterDoctors();
                filterSpecialities();
            }
            
            // Then restore visual selections when going back to step 2
            setTimeout(function() {
                if (selectedHospital) {
                    $('.hospital-filter-card').removeClass('border-green-500 bg-green-50').addClass('border-gray-200');
                    $(`.hospital-filter-card[data-hospital-id="${selectedHospital}"]`).addClass('border-green-500 bg-green-50').removeClass('border-gray-200');
                }
                
                if (selectedSpeciality) {
                    $('.specialty-filter-card').removeClass('border-purple-500 bg-purple-50').addClass('border-gray-200');
                    $(`.specialty-filter-card[data-speciality-id="${selectedSpeciality}"]`).addClass('border-purple-500 bg-purple-50').removeClass('border-gray-200');
                }
                
                if (selectedDoctor) {
                    $('.doctor-filter-card').removeClass('border-blue-500 bg-blue-50').addClass('border-gray-200');
                    $(`.doctor-filter-card[data-doctor-id="${selectedDoctor}"]`).addClass('border-blue-500 bg-blue-50').removeClass('border-gray-200');
                    $('#continueToDateTimeBtn').prop('disabled', false);
                }
                
                updateSelectionSummary();
            }, 100); // Small delay to ensure DOM is updated
        }

        let phoneVerified = false;
        let verifiedPhone = '';

        function checkPatientForAppointment() {
            const phone = $('#patient_phone').val().trim();
            
            if (!phone) {
                Swal.fire('Error', 'Please enter your phone number', 'error');
                return;
            }
            
            // Simple phone validation
            if (phone.length < 10) {
                Swal.fire('Error', 'Please enter a valid phone number', 'error');
                return;
            }

            // Hide any previous results
            $('#phoneCheckResult').hide();
            $('#patient-info').addClass('hidden');
            
            $.ajax({
                url: '<?php echo base_url('frontend/check_existing_patient'); ?>',
                method: 'POST',
                data: { phone: phone },
                dataType: 'json',
                success: function(response) {
                    if (response.exists) {
                        selectedPatient = response.patient;
                        // Don't show patient details immediately - require OTP verification first
                        $('#phoneCheckResult').html(`
                            <div class="bg-blue-50 border-blue-200">
                                <div class="text-center">
                                    <i class="fas fa-user-check text-3xl text-blue-600 mb-3"></i>
                                    <h4 class="text-xl font-semibold text-blue-800 mb-3">Patient Found!</h4>
                                    <p class="text-blue-700 mb-4">We found your patient record. Please verify your phone number to continue.</p>
                                    <button onclick="startPhoneVerificationForAppointment('${phone}')" 
                                            class="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                                        <i class="fas fa-shield-alt mr-2"></i>Verify Phone Number
                                    </button>
                                </div>
                            </div>
                        `).show();
                    } else {
                        $('#phoneCheckResult').html(`
                            <div class="bg-red-50 border-red-200">
                                <div class="text-center">
                                    <i class="fas fa-user-times text-3xl text-red-600 mb-3"></i>
                                    <h4 class="text-xl font-semibold text-red-800 mb-3">Patient Not Found</h4>
                                    <p class="text-red-700 mb-4">Please register as a patient first before booking an appointment.</p>
                                    <button onclick="window.location.href='<?php echo base_url('frontend/patient_registration'); ?>'" 
                                            class="w-full px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                                        <i class="fas fa-user-plus mr-2"></i>Register Now
                                    </button>
                                </div>
                            </div>
                        `).show();
                    }
                },
                error: function() {
                    $('#phoneCheckResult').html(`
                        <div class="bg-red-50 border-red-200">
                            <div class="text-center">
                                <i class="fas fa-exclamation-triangle text-3xl text-red-600 mb-3"></i>
                                <h4 class="text-xl font-semibold text-red-800 mb-3">Error</h4>
                                <p class="text-red-700 mb-4">Something went wrong. Please try again.</p>
                                <button onclick="$('#phoneCheckResult').hide()" 
                                        class="w-full px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
                                    Try Again
                                </button>
                            </div>
                        </div>
                    `).show();
                }
            });
        }

        function startPhoneVerificationForAppointment(phone) {
            // Show the new beautiful OTP modal immediately
            showAppointmentOtpModal(phone, 'appointment_booking');
        }

        function onPhoneVerifiedForAppointment(phone) {
            phoneVerified = true;
            verifiedPhone = phone;
            
            // Now show the patient details and allow proceeding
            $('#phoneCheckResult').html(`
                <div class="bg-green-50 border-green-200">
                    <div class="text-center">
                        <i class="fas fa-check-circle text-3xl text-green-600 mb-3"></i>
                        <h4 class="text-xl font-semibold text-green-800 mb-3">Phone Verified Successfully!</h4>
                        <p class="text-green-700 mb-4">You can now proceed to book your appointment.</p>
                    </div>
                </div>
            `);

            // Show the patient info with verified status
            $('#patient-info').html(`
                <div class="bg-green-50 border border-green-200 rounded-xl p-6">
                    <div class="flex items-center mb-4">
                        <div class="w-12 h-12 bg-green-500 rounded-full flex items-center justify-center mr-4">
                            <i class="fas fa-check text-white text-lg"></i>
                        </div>
                        <div>
                            <h4 class="text-lg font-semibold text-green-800">Patient Verified</h4>
                            <p class="text-green-600 text-sm">Ready to book appointment</p>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg p-4">
                        <h5 class="font-semibold text-gray-800 mb-3 text-center">${selectedPatient.name}</h5>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-gray-600">
                            <div class="text-center">
                                <i class="fas fa-phone text-primary mb-2"></i>
                                <p><span class="font-medium">Phone:</span> ${selectedPatient.phone}</p>
                            </div>
                            <div class="text-center">
                                <i class="fas fa-hospital text-primary mb-2"></i>
                                <p><span class="font-medium">Hospital:</span> ${selectedPatient.hospital_name}</p>
                            </div>
                        </div>
                    </div>
                    <button onclick="goToStep(2)" class="mt-4 px-6 py-2 bg-green-500 text-white rounded-lg hover:bg-green-600 transition-colors">
                        <i class="fas fa-arrow-right mr-2"></i>Continue to Book Appointment
                    </button>
                </div>
            `).removeClass('hidden');
        }

        // Initialize the unified selection when page loads
        $(document).ready(function() {
            console.log('Page loaded, starting data load...');
            
            // Add a debug check first
            $.ajax({
                url: '<?php echo base_url('frontend/test_db'); ?>',
                method: 'POST',
                dataType: 'json',
                success: function(test_data) {
                    console.log('Test DB Response:', test_data);
                    if (test_data.specialities && test_data.specialities.length > 0) {
                        console.log('✅ Specialities found:', test_data.specialities.length);
                    } else {
                        console.log('❌ No specialities found');
                    }
                    if (test_data.doctors && test_data.doctors.length > 0) {
                        console.log('✅ Doctors found:', test_data.doctors.length); 
                    } else {
                        console.log('❌ No doctors found');
                    }
                },
                error: function(xhr, status, error) {
                    console.log('❌ Test DB endpoint error:', error);
                    console.log('Response text:', xhr.responseText);
                } 
            });
            
            // Simple initial setup
            $('#specialities-filter-container').html('<p class="text-gray-500 text-center py-4">Please select a hospital first</p>');
            $('#doctors-filter-container').html('<p class="text-gray-500 text-center py-4">Please select a hospital and specialty first</p>');
            
            // Initialize date picker (disabled until doctor is selected)
            initializeDatePicker();
            
            // Test doctors endpoint on page load
            console.log('Testing doctors endpoint...');
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctors_by_hospital'); ?>',
                method: 'POST',
                data: {},
                dataType: 'json',
                success: function(data) {
                    console.log('Test doctors endpoint SUCCESS:', data);
                },
                error: function(xhr, status, error) {
                    console.log('Test doctors endpoint ERROR:', error);
                    console.log('Response:', xhr.responseText);
                }
            });
            
            // Simple direct click handlers
            setTimeout(function() {
                $('.hospital-filter-card').click(function() {
                    console.log('Hospital clicked');
                    
                    // Clear previous selections
                $('.hospital-filter-card').removeClass('border-green-500 bg-green-50').addClass('border-gray-200');
                $(this).addClass('border-green-500 bg-green-50').removeClass('border-gray-200');
                
                    selectedHospital = $(this).data('hospital-id');
                console.log('Selected hospital:', selectedHospital);
                
                    // Clear specialty and doctor areas
                selectedSpeciality = null;
                    selectedDoctor = null;
                    
                    // Load specialities for this hospital
                    console.log('About to load specialities...');
                    loadSpecialitiesForHospital(selectedHospital);
                    
                    // Also load doctors for this hospital
                    console.log('About to load doctors...');
                    loadDoctorsForHospital(selectedHospital);
                });
                
                // Handle specialty clicks (delegated since they're loaded dynamically)
            $(document).on('click', '.specialty-filter-card', function() {
                    console.log('Specialty clicked');
                    
                $('.specialty-filter-card').removeClass('border-purple-500 bg-purple-50').addClass('border-gray-200');
                $(this).addClass('border-purple-500 bg-purple-50').removeClass('border-gray-200');
                    
                selectedSpeciality = $(this).data('speciality-id');
                    console.log('Selected specialty:', selectedSpeciality);
                    
                    // Load doctors for this hospital + specialty
                    loadDoctorsForHospitalAndSpecialty(selectedHospital, selectedSpeciality);
                });
                
                // Handle doctor clicks (delegated since they're loaded dynamically)
            $(document).on('click', '.doctor-filter-card', function() {
                    console.log('Doctor clicked');
                    
                $('.doctor-filter-card').removeClass('border-blue-500 bg-blue-50').addClass('border-gray-200');
                $(this).addClass('border-blue-500 bg-blue-50').removeClass('border-gray-200');
                
                                        selectedDoctor = $(this).data('doctor-id');
                console.log('Selected doctor:', selectedDoctor);
                
                    // Load available dates for the selected doctor
                    console.log('Loading available dates for doctor:', selectedDoctor);
                    loadAvailableDates(selectedDoctor);
                    
                    $('#continueToDateTimeBtn').prop('disabled', false);
                });
            }, 500);
        });

        // Simple function to load specialities for a hospital
        function loadSpecialitiesForHospital(hospitalId) {
            console.log('Loading specialities for hospital:', hospitalId);
            
            $('#specialities-filter-container').html('<p class="text-gray-500 text-center py-4">Loading...</p>');
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_specialities_by_hospital'); ?>',
                method: 'POST',
                data: { hospital_id: hospitalId },
                dataType: 'json',
                success: function(specialities) {
                    console.log('Specialities loaded:', specialities);
                    
                    if (specialities && specialities.length > 0) {
                        let html = '';
                        specialities.forEach(function(speciality) {
                            html += `
                                <div class="specialty-filter-card bg-white border border-gray-200 rounded-lg p-3 cursor-pointer hover:border-purple-500 hover:bg-purple-50 transition-all mb-2" data-speciality-id="${speciality.id}">
                                    <div class="flex items-center">
                                        <div class="w-8 h-8 bg-purple-100 rounded-full flex items-center justify-center mr-3">
                                            <i class="fas fa-stethoscope text-purple-600 text-xs"></i>
                                        </div>
                                        <h5 class="font-medium text-gray-800 text-sm">${speciality.name}</h5>
                                    </div>
                                </div>
                            `;
                        });
                        $('#specialities-filter-container').html(html);
                    } else {
                        $('#specialities-filter-container').html('<p class="text-gray-500 text-center py-4">No specialities found for this hospital</p>');
                    }
                },
                error: function() {
                    console.log('Error loading specialities');
                    $('#specialities-filter-container').html('<p class="text-red-500 text-center py-4">Error loading specialities</p>');
                }
            });
        }

        // Simple function to load doctors for a hospital (all doctors)
        function loadDoctorsForHospital(hospitalId) {
            console.log('Loading all doctors for hospital:', hospitalId);
            console.log('AJAX URL:', '<?php echo base_url('frontend/get_doctors_by_hospital'); ?>');
            
            $('#doctors-filter-container').html('<p class="text-gray-500 text-center py-4">Loading...</p>');
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctors_by_hospital'); ?>',
                method: 'POST',
                data: { hospital_id: hospitalId },
                dataType: 'json',
                success: function(doctors) {
                    console.log('AJAX SUCCESS - All doctors loaded:', doctors);
                    console.log('Doctors array length:', doctors ? doctors.length : 'undefined');
                    
                    if (doctors && doctors.length > 0) {
                        let html = '';
                        doctors.forEach(function(doctor) {
                            html += `
                                <div class="doctor-filter-card bg-white border border-gray-200 rounded-lg p-3 cursor-pointer hover:border-blue-500 hover:bg-blue-50 transition-all mb-2" data-doctor-id="${doctor.id}">
                                    <div class="flex items-center">
                                        <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center mr-3">
                                            <i class="fas fa-user-md text-blue-600 text-xs"></i>
                                        </div>
                                        <div>
                                            <h5 class="font-medium text-gray-800 text-sm">${doctor.name}</h5>
                                            <p class="text-xs text-gray-500">${doctor.hospital_name}</p>
                                            ${doctor.speciality_name ? `<p class="text-xs text-purple-600">${doctor.speciality_name}</p>` : ''}
                                        </div>
                                    </div>
                                </div>
                            `;
                        });
                        $('#doctors-filter-container').html(html);
                } else {
                        $('#doctors-filter-container').html('<p class="text-gray-500 text-center py-4">No doctors found for this hospital</p>');
                    }
                },
                error: function(xhr, status, error) {
                    console.log('AJAX ERROR loading doctors');
                    console.log('Status:', status);
                    console.log('Error:', error);
                    console.log('Response:', xhr.responseText);
                    $('#doctors-filter-container').html('<p class="text-red-500 text-center py-4">Error loading doctors</p>');
                }
            });
        }

        // Simple function to load doctors for hospital + specialty
        function loadDoctorsForHospitalAndSpecialty(hospitalId, specialityId) {
            console.log('Loading doctors for hospital:', hospitalId, 'specialty:', specialityId);
            
            $('#doctors-filter-container').html('<p class="text-gray-500 text-center py-4">Loading...</p>');
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctors_filtered'); ?>',
                method: 'POST',
                data: { 
                    hospital_id: hospitalId,
                    speciality_id: specialityId
                },
                dataType: 'json',
                success: function(doctors) {
                    console.log('Doctors loaded:', doctors);
                    
                    if (doctors && doctors.length > 0) {
                        let html = '';
                        doctors.forEach(function(doctor) {
                            html += `
                                <div class="doctor-filter-card bg-white border border-gray-200 rounded-lg p-3 cursor-pointer hover:border-blue-500 hover:bg-blue-50 transition-all mb-2" data-doctor-id="${doctor.id}">
                                    <div class="flex items-center">
                                        <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center mr-3">
                                            <i class="fas fa-user-md text-blue-600 text-xs"></i>
                                        </div>
                                        <div>
                                            <h5 class="font-medium text-gray-800 text-sm">${doctor.name}</h5>
                                            <p class="text-xs text-gray-500">${doctor.hospital_name}</p>
                                        </div>
                                    </div>
                                </div>
                            `;
                        });
                        $('#doctors-filter-container').html(html);
                } else {
                        $('#doctors-filter-container').html('<p class="text-gray-500 text-center py-4">No doctors found for this specialty</p>');
                    }
                },
                error: function() {
                    console.log('Error loading doctors');
                    $('#doctors-filter-container').html('<p class="text-red-500 text-center py-4">Error loading doctors</p>');
                }
            });
        }

        function showInitialEmptyState() {
            // Show initial state for specialties
            $('#specialities-filter-container').html(`
                <div class="text-center py-8">
                    <div class="w-12 h-12 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-2">
                        <i class="fas fa-stethoscope text-gray-400"></i>
                    </div>
                    <p class="text-gray-500 text-sm">Select a hospital first to view specialties</p>
                </div>
            `);
            
            // Show initial state for doctors
            $('#doctors-filter-container').html(`
                <div class="text-center py-8">
                    <div class="w-12 h-12 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-2">
                        <i class="fas fa-user-md text-gray-400"></i>
                    </div>
                    <p class="text-gray-500 text-sm">Select a hospital and specialty to view doctors</p>
                </div>
            `);
        }

        function loadAllSpecialities() {
            $.ajax({
                url: '<?php echo base_url('frontend/get_specialities_by_hospital'); ?>',
                method: 'POST',
                data: {},
                dataType: 'json',
                success: function(specialities) {
                    console.log('Loaded specialities:', specialities);
                    if (specialities.error) {
                        console.error('Backend error:', specialities.error);
                        $('#specialities-filter-container').html(`
                            <div class="text-center py-8">
                                <div class="w-12 h-12 bg-red-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                    <i class="fas fa-exclamation-triangle text-red-400"></i>
                                </div>
                                <p class="text-red-500 text-sm">Backend error: ${specialities.error}</p>
                                <p class="text-gray-500 text-xs mt-2">Check browser console for details</p>
                            </div>
                        `);
                        return;
                    }
                    
                    if (!Array.isArray(specialities) || specialities.length === 0) {
                        console.log('No specialities found or invalid response format');
                        $('#specialities-filter-container').html(`
                            <div class="text-center py-8">
                                <div class="w-12 h-12 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                    <i class="fas fa-stethoscope text-gray-400"></i>
                                </div>
                                <p class="text-gray-500 text-sm">No specialities available</p>
                            </div>
                        `);
                        return;
                    }
                    
                    allSpecialities = specialities;
                    renderSpecialities(specialities);
                },
                error: function(xhr, status, error) {
                    console.error('Error loading specialities:', error);
                    console.error('Response:', xhr.responseText);
                    $('#specialities-filter-container').html(`
                        <div class="text-center py-8">
                            <div class="w-12 h-12 bg-red-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                <i class="fas fa-exclamation-triangle text-red-400"></i>
                            </div>
                            <p class="text-red-500 text-sm">Error loading specialities. Please refresh the page.</p>
                        </div>
                    `);
                }
            });
        }

        function loadSpecialitiesByHospital(hospitalId) {
            console.log('Loading specialities for hospital:', hospitalId);
            console.log('Making AJAX call to:', '<?php echo base_url('frontend/get_specialities_by_hospital'); ?>');
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_specialities_by_hospital'); ?>',
                method: 'POST',
                data: { hospital_id: hospitalId },
                dataType: 'json',
                success: function(specialities) {
                    console.log('Loaded specialities by hospital:', specialities);
                    if (specialities.error) {
                        console.error('Backend error:', specialities.error);
                        $('#specialities-filter-container').html(`
                            <div class="text-center py-8">
                                <div class="w-12 h-12 bg-red-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                    <i class="fas fa-exclamation-triangle text-red-400"></i>
                                </div>
                                <p class="text-red-500 text-sm">Backend error: ${specialities.error}</p>
                            </div>
                        `);
                        return;
                    }
                    allSpecialities = specialities;
                    renderSpecialities(specialities);
                },
                error: function(xhr, status, error) {
                    console.error('Error loading specialities by hospital:', error);
                    console.error('Status:', status);
                    console.error('XHR:', xhr);
                    console.error('Response:', xhr.responseText);
                    $('#specialities-filter-container').html(`
                        <div class="text-center py-8">
                            <div class="w-12 h-12 bg-red-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                <i class="fas fa-exclamation-triangle text-red-400"></i>
                            </div>
                            <p class="text-red-500 text-sm">Error loading specialities. Please refresh the page.</p>
                        </div>
                    `);
                }
            });
        }

        function loadAllDoctors() {
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctors_filtered'); ?>',
                method: 'POST',
                data: {},
                dataType: 'json',
                success: function(doctors) {
                    console.log('Loaded doctors:', doctors);
                    if (doctors.error) {
                        console.error('Backend error:', doctors.error);
                        $('#doctors-filter-container').html(`
                            <div class="text-center py-8">
                                <div class="w-12 h-12 bg-red-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                    <i class="fas fa-exclamation-triangle text-red-400"></i>
                                </div>
                                <p class="text-red-500 text-sm">Backend error: ${doctors.error}</p>
                                <p class="text-gray-500 text-xs mt-2">Check browser console for details</p>
                            </div>
                        `);
                        return;
                    }
                    
                    if (!Array.isArray(doctors) || doctors.length === 0) {
                        console.log('No doctors found or invalid response format');
                        $('#doctors-filter-container').html(`
                            <div class="text-center py-8">
                                <div class="w-12 h-12 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                    <i class="fas fa-user-md text-gray-400"></i>
                                </div>
                                <p class="text-gray-500 text-sm">No doctors available</p>
                            </div>
                        `);
                        return;
                    }
                    allDoctors = doctors;
                    renderDoctors(doctors);
                },
                error: function(xhr, status, error) {
                    console.error('Error loading doctors:', error);
                    console.error('Response:', xhr.responseText);
                    $('#doctors-filter-container').html(`
                        <div class="text-center py-8">
                            <div class="w-12 h-12 bg-red-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                <i class="fas fa-exclamation-triangle text-red-400"></i>
                            </div>
                            <p class="text-red-500 text-sm">Error loading doctors. Please refresh the page.</p>
                        </div>
                    `);
                }
            });
        }

        function loadDoctorsByHospital(hospitalId) {
            console.log('Loading doctors for hospital:', hospitalId);
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctors_by_hospital'); ?>',
                method: 'POST',
                data: { hospital_id: hospitalId },
                dataType: 'json',
                success: function(doctors) {
                    console.log('Loaded doctors by hospital:', doctors);
                    if (doctors.error) {
                        console.error('Backend error:', doctors.error);
                        return;
                    }
                    allDoctors = doctors;
                    renderDoctors(doctors);
                },
                error: function(xhr, status, error) {
                    console.error('Error loading doctors by hospital:', error);
                    console.error('Response:', xhr.responseText);
                }
            });
        }

        function loadDoctorsBySpeciality(specialityId) {
            console.log('Loading doctors for speciality:', specialityId);
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctors_by_speciality'); ?>',
                method: 'POST',
                data: { 
                    speciality_id: specialityId,
                    hospital_id: selectedHospital // Include hospital filter if selected
                },
                dataType: 'json',
                success: function(doctors) {
                    console.log('Loaded doctors by speciality:', doctors);
                    if (doctors.error) {
                        console.error('Backend error:', doctors.error);
                        return;
                    }
                    allDoctors = doctors;
                    renderDoctors(doctors);
                },
                error: function(xhr, status, error) {
                    console.error('Error loading doctors by speciality:', error);
                    console.error('Response:', xhr.responseText);
                }
            });
        }

        function loadDoctorsByHospitalAndSpeciality(hospitalId, specialityId) {
            console.log('Loading doctors for hospital and speciality:', hospitalId, specialityId);
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctors_filtered'); ?>',
                method: 'POST',
                data: { 
                    hospital_id: hospitalId,
                    speciality_id: specialityId
                },
                dataType: 'json',
                success: function(doctors) {
                    console.log('Loaded doctors by hospital and speciality:', doctors);
                    if (doctors.error) {
                        console.error('Backend error:', doctors.error);
                        return;
                    }
                    allDoctors = doctors;
                    renderDoctors(doctors);
                },
                error: function(xhr, status, error) {
                    console.error('Error loading doctors by hospital and speciality:', error);
                    console.error('Response:', xhr.responseText);
                }
            });
        }

        function renderSpecialities(specialities) {
                    let html = '';
                    specialities.forEach(function(speciality) {
                const isSelected = selectedSpeciality == speciality.id;
                        html += `
                    <div class="specialty-filter-card bg-white border ${isSelected ? 'border-purple-500 bg-purple-50' : 'border-gray-200'} rounded-lg p-3 cursor-pointer hover:border-purple-500 hover:bg-purple-50 transition-all"
                                 data-speciality-id="${speciality.id}">
                        <div class="flex items-center">
                            <div class="w-8 h-8 bg-purple-100 rounded-full flex items-center justify-center mr-3">
                                <i class="fas fa-stethoscope text-purple-600 text-xs"></i>
                                </div>
                            <h5 class="font-medium text-gray-800 text-sm">${speciality.name}</h5>
                        </div>
                            </div>
                        `;
                    });
            $('#specialities-filter-container').html(html);
        }

        function renderDoctors(doctors) {
                    let html = '';
                    doctors.forEach(function(doctor) {
                const isSelected = selectedDoctor == doctor.id;
                        html += `
                    <div class="doctor-filter-card bg-white border ${isSelected ? 'border-blue-500 bg-blue-50' : 'border-gray-200'} rounded-lg p-3 cursor-pointer hover:border-blue-500 hover:bg-blue-50 transition-all"
                         data-doctor-id="${doctor.id}" data-hospital-id="${doctor.hospital_id}" data-speciality-id="${doctor.speciality_id}">
                                <div class="flex items-center">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center mr-3">
                                <i class="fas fa-user-md text-blue-600 text-xs"></i>
                                    </div>
                                    <div class="flex-1">
                                <h5 class="font-medium text-gray-800 text-sm">${doctor.name}</h5>
                                <p class="text-xs text-gray-500">${doctor.hospital_name}</p>
                                ${doctor.speciality_name ? `<p class="text-xs text-purple-600">${doctor.speciality_name}</p>` : ''}
                                    </div>
                                </div>
                            </div>
                        `;
                    });
                    
                    if (doctors.length === 0) {
                        html = `
                    <div class="text-center py-8">
                        <div class="w-12 h-12 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-2">
                            <i class="fas fa-user-md text-gray-400"></i>
                                </div>
                        <p class="text-gray-500 text-sm">No doctors available</p>
                            </div>
                        `;
                    }
                    
            $('#doctors-filter-container').html(html);
        }

        function filterDoctors() {
            console.log('Filtering doctors with:', {
                selectedHospital: selectedHospital,
                selectedSpeciality: selectedSpeciality,
                totalDoctors: allDoctors.length
            });
            
            const filteredDoctors = allDoctors.filter(doctor => {
                let matches = true;
                console.log('Checking doctor:', doctor.name, {
                    doctor_hospital_id: doctor.hospital_id,
                    doctor_speciality_id: doctor.speciality_id,
                    selectedHospital: selectedHospital,
                    selectedSpeciality: selectedSpeciality
                });
                
                // Convert to strings for comparison to handle type mismatches
                if (selectedHospital && String(doctor.hospital_id) !== String(selectedHospital)) matches = false;
                if (selectedSpeciality && String(doctor.speciality_id) !== String(selectedSpeciality)) matches = false;
                
                console.log('Doctor matches:', matches);
                return matches;
            });
            
            console.log('Filtered doctors:', filteredDoctors.length, 'results');
            renderDoctors(filteredDoctors);
        }

        function filterSpecialities() {
            if (!selectedHospital) {
                renderSpecialities(allSpecialities);
                return;
            }
            
            // Filter specialties based on available doctors in selected hospital
            const hospitalDoctors = allDoctors.filter(doctor => doctor.hospital_id == selectedHospital);
            const availableSpecialityIds = [...new Set(hospitalDoctors.map(doctor => doctor.speciality_id))].filter(id => id);
            const filteredSpecialities = allSpecialities.filter(speciality => availableSpecialityIds.includes(speciality.id));
            
            renderSpecialities(filteredSpecialities);
        }

        function searchDoctors(searchTerm) {
            $.ajax({
                url: '<?php echo base_url('frontend/search_doctors'); ?>',
                method: 'POST',
                data: { search_term: searchTerm },
                dataType: 'json',
                success: function(doctors) {
                    console.log('Search results for "' + searchTerm + '":', doctors);
                    
                    if (doctors.error) {
                        console.error('Search error:', doctors.error);
                        $('#doctors-filter-container').html(`
                            <div class="text-center py-4">
                                <p class="text-red-500 text-sm">${doctors.error}</p>
                            </div>
                        `);
                        return;
                    }
                    
                    if (!Array.isArray(doctors) || doctors.length === 0) {
                        $('#doctors-filter-container').html(`
                            <div class="text-center py-8">
                                <div class="w-12 h-12 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-2">
                                    <i class="fas fa-search text-gray-400"></i>
                                </div>
                                <p class="text-gray-500 text-sm">No doctors found matching "${searchTerm}"</p>
                                <p class="text-gray-400 text-xs mt-1">Try a different search term</p>
                            </div>
                        `);
                        return;
                    }
                    
                    // Store search results and render
                    allDoctors = doctors;
                    renderDoctors(doctors);
                },
                error: function(xhr, status, error) {
                    console.error('Search request failed:', error);
                    console.error('Response:', xhr.responseText);
                    $('#doctors-filter-container').html(`
                        <div class="text-center py-4">
                            <p class="text-red-500 text-sm">Search failed. Please try again.</p>
                        </div>
                    `);
                }
            });
        }

        function updateSelectionSummary() {
            const summaryDiv = $('#selection-summary');
            let hasSelection = false;
            
            if (selectedHospital) {
                const hospitalName = $(`.hospital-filter-card[data-hospital-id="${selectedHospital}"]`).find('h5').text();
                $('#selected-hospital-name').text(hospitalName);
                $('#selected-hospital-info').removeClass('hidden');
                hasSelection = true;
            } else {
                $('#selected-hospital-info').addClass('hidden');
            }
            
            if (selectedSpeciality) {
                const specialityName = $(`.specialty-filter-card[data-speciality-id="${selectedSpeciality}"]`).find('h5').text();
                $('#selected-specialty-name').text(specialityName);
                $('#selected-specialty-info').removeClass('hidden');
                hasSelection = true;
            } else {
                $('#selected-specialty-info').addClass('hidden');
            }
            
            if (selectedDoctor) {
                const doctorName = $(`.doctor-filter-card[data-doctor-id="${selectedDoctor}"]`).find('h5').text();
                $('#selected-doctor-name').text(doctorName);
                $('#selected-doctor-info').removeClass('hidden');
                hasSelection = true;
                $('#continueToDateTimeBtn').prop('disabled', false);
            } else {
                $('#selected-doctor-info').addClass('hidden');
                $('#continueToDateTimeBtn').prop('disabled', true);
            }
            
            if (hasSelection) {
                summaryDiv.removeClass('hidden');
            } else {
                summaryDiv.addClass('hidden');
            }
        }

        function continueToDateTime() {
            if (!selectedDoctor) return;
            goToStep(3);
        }

        function continueToPayment() {
            if (!selectedTimeSlot) {
                Swal.fire('Error', 'Please select a date and time slot', 'error');
                return;
            }
            if (!selectedVisitType) {
                Swal.fire('Error', 'Please select a visit type', 'error');
                return;
            }
            
            // Always go to payment step - the configurePaymentOptions will handle what to show
            goToStep(4);
        }

        function loadDoctorDetails() {
            if (!selectedDoctor) return;
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctor_details'); ?>',
                method: 'POST',
                data: { doctor_id: selectedDoctor },
                dataType: 'json',
                success: function(response) {
                    if (response.success && response.doctor) {
                        const doctor = response.doctor;
                        
                        // Update doctor details in the UI
                        $('#doctor-detail-name').text(doctor.name || 'Doctor Name');
                        $('#doctor-detail-specialty').text(doctor.speciality_name || 'General Practice');
                        $('#doctor-detail-hospital').text(doctor.hospital_name || 'Hospital Network');
                        $('#doctor-detail-qualification').text(doctor.qualification || 'Qualified Medical Professional');
                        
                        // Format consultation fee
                        const fee = doctor.price ? `$${doctor.price}` : 'Contact for pricing';
                        $('#doctor-detail-fee').text(fee);
                        
                        // Store doctor's payment setting
                        doctorPaymentSetting = doctor.online_payment_enabled || 'disabled';
                        
                        // Update button text based on payment setting
                        updateContinueButtonText();
                    }
                },
                error: function() {
                    console.log('Error loading doctor details');
                }
            });
        }

        function loadVisitTypes() {
            if (!selectedDoctor) return;
            
            console.log('Loading visit types for doctor:', selectedDoctor);
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_doctor_visit_types'); ?>',
                method: 'POST',
                data: { doctor_id: selectedDoctor },
                dataType: 'json',
                success: function(response) {
                    console.log('Visit types response:', response);
                    if (response.success) {
                        let options = '<option value="">Select visit type...</option>';
                        response.visit_types.forEach(function(visit) {
                            options += `<option value="${visit.id}">${visit.visit_description}</option>`;
                        });
                        $('#visit_type').html(options);
                        console.log('Visit types loaded:', response.visit_types.length, 'types');
                    } else {
                        console.log('No visit types found or error:', response.message);
                        Swal.fire('Info', 'No visit types available for this doctor', 'info');
                    }
                },
                error: function(xhr, status, error) {
                    console.log('AJAX Error loading visit types:', xhr.responseText);
                    console.log('Status:', status);
                    console.log('Error:', error);
                }
            });
        }

        // Visit type change handler
        $(document).on('change', '#visit_type', function() {
            const visitTypeId = $(this).val();
            selectedVisitType = visitTypeId;
            
            console.log('Visit type selected:', visitTypeId);
            
            if (visitTypeId) {
                // Load visit charges
                $.ajax({
                    url: '<?php echo base_url('frontend/get_visit_charges'); ?>',
                    method: 'POST',
                    data: { visit_type_id: visitTypeId },
                dataType: 'json',
                    success: function(response) {
                        console.log('Visit charges response:', response);
                        if (response.success) {
                            $('#visit_charges').val('$' + response.visit_charges);
                            calculateGrandTotal();
                            // Enable continue button if both time slot and visit type are selected
                            if (selectedTimeSlot) {
                                $('#continueToPaymentBtn').prop('disabled', false);
                            }
                        } else {
                            console.log('Error in response:', response.message);
                            Swal.fire('Error', response.message || 'Failed to load visit charges', 'error');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('AJAX Error:', xhr.responseText);
                        console.log('Status:', status);
                        console.log('Error:', error);
                        Swal.fire('Error', 'Failed to load visit charges. Please try again.', 'error');
                    }
                });
            } else {
                $('#visit_charges').val('');
                $('#grand_total').val('');
                selectedVisitType = null;
                $('#continueToPaymentBtn').prop('disabled', true);
            }
        });

        // Discount calculation
        $(document).on('input', '#discount', function() {
            calculateGrandTotal();
        });

        function calculateGrandTotal() {
            const visitCharges = parseFloat($('#visit_charges').val().replace('$', '')) || 0;
            const discount = parseFloat($('#discount').val()) || 0;
            const grandTotal = visitCharges - discount;
            $('#grand_total').val('$' + grandTotal.toFixed(2));
        }

        function loadAppointmentSummary() {
            // Load appointment summary for payment step
            if (selectedPatient && selectedDoctor) {
                $('#payment-patient-name').text(selectedPatient.name);
                $('#payment-doctor-name').text($('#doctor-detail-name').text());
                $('#payment-specialty').text($('#doctor-detail-specialty').text());
                $('#payment-hospital').text($('#doctor-detail-hospital').text());
                $('#payment-date').text($('#appointment_date').val());
                $('#payment-time').text(selectedTimeSlot);
                $('#payment-amount').text($('#grand_total').val());
                
                // Configure payment options based on doctor's payment setting
                configurePaymentOptions();
            }
        }

                function configurePaymentOptions() {
            // Reset all payment info divs
            $('#card-payment-form').addClass('hidden');
            $('#pay-later-info').addClass('hidden');
            $('#optional-payment-info').addClass('hidden');
            
            // Remove any existing info messages
            $('#required-payment-info, #disabled-payment-info').remove();
            
            // Clear any previous selections
            $('.payment-method').removeClass('border-blue-500 bg-blue-50').addClass('border-gray-200');
            selectedPaymentMethod = null;
            
            if (doctorPaymentSetting === 'required') {
                // Payment is required - show online payment options only
                $('.payment-method[data-method="stripe"]').show();
                $('.payment-method[data-method="pay_later"]').hide();
                
                // Auto-select first available payment method (Stripe)
                setTimeout(() => {
                    $('.payment-method[data-method="stripe"]').click();
                }, 100);
                
                // Show info message
                $('.payment-method').parent().parent().prepend(`
                    <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6" id="required-payment-info">
                        <p class="text-red-800">
                            <i class="fas fa-exclamation-triangle mr-2"></i>
                            This doctor requires online payment to confirm your appointment.
                        </p>
                    </div>
                `);
                
            } else if (doctorPaymentSetting === 'optional') {
                // Payment is optional - show all options
                $('.payment-method[data-method="stripe"]').show();
                $('.payment-method[data-method="pay_later"]').show();
                
                // Show optional payment info
                $('#optional-payment-info').removeClass('hidden');
                
            } else {
                // Book without payment - show only pay later option
                $('.payment-method[data-method="stripe"]').hide();
                $('.payment-method[data-method="pay_later"]').show();
                
                // Auto-select pay later
                setTimeout(() => {
                    $('.payment-method[data-method="pay_later"]').click();
                }, 100);
                
                // Show info message
                $('.payment-method').parent().parent().prepend(`
                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6" id="disabled-payment-info">
                        <p class="text-blue-800">
                            <i class="fas fa-info-circle mr-2"></i>
                            This doctor allows booking without advance payment. You can pay during your visit.
                        </p>
                    </div>
                `);
            }
        }

        function updateContinueButtonText() {
            const btnText = $('#continue-btn-text');
            const btnIcon = $('#continueToPaymentBtn i');
            
            // Always show payment step, but button text changes based on setting
            if (doctorPaymentSetting === 'disabled') {
                btnText.text('Continue to Booking');
                btnIcon.removeClass('fa-calendar-check').addClass('fa-arrow-right');
            } else if (doctorPaymentSetting === 'required') {
                btnText.text('Continue to Payment');
                btnIcon.removeClass('fa-calendar-check').addClass('fa-arrow-right');
            } else {
                btnText.text('Continue to Payment');
                btnIcon.removeClass('fa-calendar-check').addClass('fa-arrow-right');
            }
            
            // Always show payment step indicator since we always go to step 4
            $('#step-indicator-4').parent().show();
            $('#line-3').show();
        }

        function processAppointmentDirectly() {
            const appointmentData = {
                patient_id: selectedPatient.id,
                doctor_id: selectedDoctor,
                hospital_id: selectedHospital,
                date: $('#appointment_date').val(),
                time_slot: selectedTimeSlot,
                problem: $('#appointment_problem').val(),
                visit_type: selectedVisitType,
                visit_charges: parseFloat($('#visit_charges').val().replace('$', '')),
                discount: parseFloat($('#discount').val()) || 0,
                grand_total: parseFloat($('#grand_total').val().replace('$', '')),
                payment_method: 'pay_later',
                booking_source: 'online',
            };

            Swal.fire({
                title: 'Booking Appointment...',
                html: 'Creating your appointment.',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            
            $.ajax({
                url: '<?php echo base_url('frontend/book_appointment_with_payment'); ?>',
                method: 'POST',
                data: appointmentData,
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        // Beautiful appointment success message
                        Swal.fire({
                            title: 'Appointment Booked Successfully!',
                            html: `
                                <div class="text-center">
                                    <i class="fas fa-calendar-check text-green-600 text-4xl mb-4"></i>
                                    <p class="text-lg mb-2">Your appointment has been scheduled successfully!</p>
                                    <div class="bg-green-50 p-4 rounded-lg mb-4">
                                        <p class="font-semibold text-green-800">📅 Appointment Details:</p>
                                        <p class="text-green-700">Appointment ID: <strong>#${response.appointment_id || 'Generated'}</strong></p>
                                        <p class="text-green-700">Status: <strong>Confirmed</strong></p>
                                        <p class="text-sm text-green-600 mt-2">💰 No payment required - Pay at hospital</p>
                                    </div>
                                    <div class="bg-blue-50 p-4 rounded-lg mb-4">
                                        <p class="font-semibold text-blue-800">📋 Next Steps:</p>
                                        <p class="text-blue-700 text-sm">• Arrive 15 minutes before your appointment</p>
                                        <p class="text-blue-700 text-sm">• Bring a valid ID and any medical records</p>
                                        <p class="text-blue-700 text-sm">• Payment can be made at the hospital</p>
                                    </div>
                                    <p class="text-gray-600">View your appointment details and get directions!</p>
                                </div>
                            `,
                            icon: 'success',
                            showCancelButton: true,
                            confirmButtonText: '<i class="fas fa-eye mr-2"></i>View Details',
                            cancelButtonText: '<i class="fas fa-home mr-2"></i>Go to Home',
                            confirmButtonColor: '#059669',
                            cancelButtonColor: '#6b7280'
                        }).then((result) => {
                            if (result.isConfirmed) {
                                window.location.href = '<?php echo base_url('frontend/appointment_confirmation'); ?>/' + response.appointment_id;
                            } else {
                                window.location.href = '<?php echo base_url('frontend/public_portal'); ?>';
                            }
                        });
                    } else {
                        Swal.fire('Error', response.message || 'Booking failed. Please try again.', 'error');
                    }
                },
                error: function() {
                    Swal.fire('Error', 'Something went wrong. Please try again.', 'error');
                }
            });
        }

        // Payment method selection
        $(document).on('click', '.payment-method', function() {
            $('.payment-method').removeClass('border-blue-500 bg-blue-50').addClass('border-gray-200');
            $(this).addClass('border-blue-500 bg-blue-50').removeClass('border-gray-200');
            selectedPaymentMethod = $(this).data('method');
            
            // Hide all payment forms
            $('#card-payment-form').addClass('hidden');
            $('#paypal-payment-info').addClass('hidden');
            $('#pay-later-info').addClass('hidden');
            
            // Show relevant form
            if (selectedPaymentMethod === 'stripe') {
                $('#card-payment-form').removeClass('hidden');
            } else if (selectedPaymentMethod === 'pay_later') {
                $('#pay-later-info').removeClass('hidden');
            }
            
            $('#processPaymentBtn').prop('disabled', false);
        });

        function processAppointmentPayment() {
            if (!selectedPaymentMethod) {
                Swal.fire('Error', 'Please select a payment method', 'error');
                return;
            }

            if (selectedPaymentMethod === 'stripe') {
                // Validate card details
                const cardNumber = $('#card_number').val();
                const cardholderName = $('#cardholder_name').val();
                const expiryDate = $('#expiry_date').val();
                const cvv = $('#cvv').val();

                if (!cardNumber || !cardholderName || !expiryDate || !cvv) {
                    Swal.fire('Error', 'Please fill in all card details', 'error');
                    return;
                }
                
            }
            
            // Create appointment first
            createAppointmentWithPayment();
        }

        function createAppointmentWithPayment() {
            const appointmentData = {
                patient_id: selectedPatient.id,
                doctor_id: selectedDoctor,
                hospital_id: selectedHospital,
                date: $('#appointment_date').val(),
                time_slot: selectedTimeSlot,
                problem: $('#appointment_problem').val(),
                visit_type: selectedVisitType,
                visit_charges: parseFloat($('#visit_charges').val().replace('$', '')),
                discount: parseFloat($('#discount').val()) || 0,
                grand_total: parseFloat($('#grand_total').val().replace('$', '')),
                payment_method: selectedPaymentMethod
            };

            Swal.fire({
                title: 'Processing...',
                html: 'Creating your appointment and processing payment.',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            $.ajax({
                url: '<?php echo base_url('frontend/book_appointment_with_payment'); ?>',
                method: 'POST',
                data: appointmentData,
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        if (selectedPaymentMethod === 'pay_later') {
                            // No payment required
                            Swal.fire({
                                title: 'Appointment Booked!',
                                text: 'Your appointment has been scheduled. You can pay at the hospital.',
                                icon: 'success',
                                confirmButtonText: 'View Details'
                            }).then(() => {
                                window.location.href = '<?php echo base_url('frontend/appointment_confirmation'); ?>/' + response.appointment_id;
                            });
                        } else {
                            // Process payment
                            processPaymentNow(response);
                        }
                    } else {
                        Swal.fire('Error', response.message || 'Booking failed. Please try again.', 'error');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', xhr.responseText);
                    console.error('Status:', status);
                    console.error('Error:', error);
                    
                    let errorMessage = 'Something went wrong. Please try again.';
                    if (xhr.responseText) {
                        try {
                            const errorResponse = JSON.parse(xhr.responseText);
                            errorMessage = errorResponse.message || errorMessage;
                        } catch (e) {
                            errorMessage = 'Server Error: ' + xhr.responseText.substring(0, 200);
                        }
                    }
                    
                    Swal.fire('Error', errorMessage, 'error');
                }
            });
        }

        function processPaymentNow(appointmentResponse) {
            const paymentData = {
                payment_id: appointmentResponse.payment_id,
                appointment_id: appointmentResponse.appointment_id,
                amount: appointmentResponse.amount,
                payment_method: selectedPaymentMethod
            };

            if (selectedPaymentMethod === 'stripe') {
                paymentData.card_number = $('#card_number').val();
                paymentData.cardholder_name = $('#cardholder_name').val();
                paymentData.expire_date = $('#expiry_date').val();
                paymentData.cvv = $('#cvv').val();
                paymentData.token = 'demo_token'; // In real implementation, use Stripe.js to get token
            }

            $.ajax({
                url: '<?php echo base_url('frontend/process_payment'); ?>',
                method: 'POST',
                data: paymentData,
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        // Beautiful payment success message
                        Swal.fire({
                            title: 'Payment Successful!',
                            html: `
                                <div class="text-center">
                                    <i class="fas fa-credit-card text-green-600 text-4xl mb-4"></i>
                                    <p class="text-lg mb-2">Your payment has been processed successfully!</p>
                                    <div class="bg-green-50 p-4 rounded-lg mb-4">
                                        <p class="font-semibold text-green-800">💳 Payment Details:</p>
                                        <p class="text-green-700">Transaction ID: <strong>#${response.transaction_id || 'Processed'}</strong></p>
                                        <p class="text-green-700">Status: <strong>Paid</strong></p>
                                        <p class="text-green-700">Appointment: <strong>Confirmed</strong></p>
                                    </div>
                                    <div class="bg-blue-50 p-4 rounded-lg mb-4">
                                        <p class="font-semibold text-blue-800">📋 Next Steps:</p>
                                        <p class="text-blue-700 text-sm">• Your appointment is fully confirmed</p>
                                        <p class="text-blue-700 text-sm">• Arrive 15 minutes before your appointment</p>
                                        <p class="text-blue-700 text-sm">• Bring a valid ID and any medical records</p>
                                        <p class="text-blue-700 text-sm">• Receipt will be emailed to you</p>
                                    </div>
                                    <p class="text-gray-600">View your appointment details and download receipt!</p>
                                </div>
                            `,
                            icon: 'success',
                            showCancelButton: true,
                            confirmButtonText: '<i class="fas fa-eye mr-2"></i>View Details',
                            cancelButtonText: '<i class="fas fa-home mr-2"></i>Go to Home',
                            confirmButtonColor: '#059669',
                            cancelButtonColor: '#6b7280'
                        }).then((result) => {
                            if (result.isConfirmed) {
                                window.location.href = '<?php echo base_url('frontend/appointment_confirmation'); ?>/' + appointmentResponse.appointment_id;
                            } else {
                                window.location.href = '<?php echo base_url('frontend/public_portal'); ?>';
                            }
                        });
                    } else {
                        Swal.fire('Payment Failed', response.message || 'Payment processing failed. Please try again.', 'error');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Payment AJAX Error:', xhr.responseText);
                    console.error('Status:', status);
                    console.error('Error:', error);
                    
                    let errorMessage = 'Payment processing failed. Please try again.';
                    if (xhr.responseText) {
                        try {
                            const errorResponse = JSON.parse(xhr.responseText);
                            errorMessage = errorResponse.message || errorMessage;
                        } catch (e) {
                            errorMessage = 'Payment Error: ' + xhr.responseText.substring(0, 200);
                        }
                    }
                    
                    Swal.fire('Error', errorMessage, 'error');
                }
            });
        }

        // Function to load available dates for selected doctor
        function loadAvailableDates(doctorId) {
            console.log('loadAvailableDates called with doctor ID:', doctorId);
            if (!doctorId) {
                console.log('No doctor ID provided, returning');
                return;
            }
            
            // Show loading message
            $('#available-dates-info').removeClass('hidden');
            $('#available-dates-text').text('Loading available dates...');
            console.log('Loading available dates for doctor:', doctorId);
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_available_dates'); ?>',
                method: 'POST',
                data: { doctor_id: doctorId },
                dataType: 'json',
                success: function(response) {
                    console.log('Available dates AJAX response:', response);
                    if (response.success) {
                        availableDates = response.available_dates;
                        console.log('Available dates loaded:', availableDates);
                        // Reset the date input and clear any selected date
                        if (datePicker) {
                            datePicker.clear();
                        } else {
                            $('#appointment_date').val('');
                        }
                        selectedTimeSlot = null;
                        $('#time-slots-container').html(`
                            <div class="text-center text-gray-500 py-8">
                                <i class="fas fa-clock text-4xl mb-4"></i>
                                <p>Please select a date to view available time slots</p>
                            </div>
                        `);
                        $('#continueToPaymentBtn').prop('disabled', true);
                        
                        // Update the date input to restrict to available dates
                        updateDateInput();
                        
                        // Update placeholder text and remove disabled styling
                        $('#appointment_date').attr('placeholder', 'Click to select an available date').removeClass('no-doctor');
                        
                        // Show available dates information
                        displayAvailableDatesInfo();
                    } else {
                        console.error('Failed to load available dates:', response.message);
                        availableDates = [];
                        $('#available-dates-info').addClass('hidden');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error loading available dates:', error);
                    console.error('Status:', status);
                    console.error('Response:', xhr.responseText);
                    availableDates = [];
                    $('#available-dates-info').addClass('hidden');
                }
            });
        }

        // Function to initialize date picker when no doctor is selected
        function initializeDatePicker() {
            if (datePicker) {
                datePicker.destroy();
            }
            
            const today = new Date();
            const maxDate = new Date();
            maxDate.setDate(maxDate.getDate() + 90);
            
            datePicker = flatpickr('#appointment_date', {
                dateFormat: 'Y-m-d',
                minDate: today,
                maxDate: maxDate,
                disable: [
                    // Disable all dates when no doctor is selected
                    function(date) {
                        return true; // Disable all dates
                    }
                ],
                onChange: function(selectedDates, dateStr, instance) {
                    if (dateStr && !selectedDoctor) {
                        datePicker.clear();
                        Swal.fire({
                            icon: 'info',
                            title: 'Select Doctor First',
                            text: 'Please select a doctor first to see their available dates.',
                            timer: 3000,
                            showConfirmButton: false
                        });
                    }
                },
                onDayCreate: function(dObj, dStr, fp, dayElem) {
                    dayElem.title = 'Please select a doctor first';
                },
                locale: {
                    firstDayOfWeek: 1 // Start week on Monday
                }
            });
        }

        // Function to display available dates information
        function displayAvailableDatesInfo() {
            const infoElement = $('#available-dates-info');
            const textElement = $('#available-dates-text');
            
            if (availableDates.length === 0) {
                textElement.html('<span class="text-red-600">No available dates found for this doctor in the next 90 days. Please try selecting a different doctor.</span>');
                infoElement.removeClass('hidden');
                $('#appointment_date').attr('placeholder', 'No available dates for this doctor').addClass('no-doctor');
                return;
            }
            
            // Format dates nicely
            const formatDate = (dateStr) => {
                const date = new Date(dateStr);
                const options = { 
                    weekday: 'short', 
                    month: 'short', 
                    day: 'numeric'
                };
                return date.toLocaleDateString('en-US', options);
            };
            
            // Show first few available dates
            const nextFewDates = availableDates.slice(0, 5);
            const formattedDates = nextFewDates.map(formatDate);
            
            let message = `Available dates (click to select): ${formattedDates.join(', ')}`;
            if (availableDates.length > 5) {
                message += ` and ${availableDates.length - 5} more available...`;
            }
            
            textElement.text(message);
            infoElement.removeClass('hidden');
        }

        // Function to update date input with restrictions using Flatpickr
        function updateDateInput() {
            // Destroy existing date picker if it exists
            if (datePicker) {
                datePicker.destroy();
            }
            
            // Calculate min and max dates
            const today = new Date();
            const maxDate = new Date();
            maxDate.setDate(maxDate.getDate() + 90);
            
            // Initialize Flatpickr with available dates
            datePicker = flatpickr('#appointment_date', {
                dateFormat: 'Y-m-d',
                minDate: today,
                maxDate: maxDate,
                disable: [
                    // Disable all dates by default
                    function(date) {
                        const year = date.getFullYear();
                        const month = String(date.getMonth() + 1).padStart(2, '0');
                        const day = String(date.getDate()).padStart(2, '0');
                        const dateStr = `${year}-${month}-${day}`;
                        return !availableDates.includes(dateStr);
                    }
                ],
                enable: availableDates.map(dateStr => {
                    const parts = dateStr.split('-');
                    return new Date(parseInt(parts[0]), parseInt(parts[1]) - 1, parseInt(parts[2]));
                }),
                onChange: function(selectedDates, dateStr, instance) {
                    if (dateStr && selectedDoctor) {
                        loadTimeSlots(dateStr);
                    }
                },
                onDayCreate: function(dObj, dStr, fp, dayElem) {
                    // Use local date string to avoid timezone issues
                    const year = dayElem.dateObj.getFullYear();
                    const month = String(dayElem.dateObj.getMonth() + 1).padStart(2, '0');
                    const day = String(dayElem.dateObj.getDate()).padStart(2, '0');
                    const dateStr = `${year}-${month}-${day}`;
                    
                    console.log('Checking date:', dateStr, 'Available dates:', availableDates);
                    if (availableDates.includes(dateStr)) {
                        dayElem.classList.add('available');
                        dayElem.title = 'Available for appointment';
                        console.log('Date marked as available:', dateStr);
                    } else {
                        dayElem.title = 'Not available';
                    }
                },
                locale: {
                    firstDayOfWeek: 1 // Start week on Monday
                }
            });
        }

        // Date selection and time slot loading is now handled by Flatpickr onChange event
        // We can keep this as a fallback for manual input validation
        $('#appointment_date').on('change', function() {
            const selectedDate = $(this).val();
            if (selectedDate && !selectedDoctor) {
                // If date is selected but no doctor, remind user to select doctor first
                if (datePicker) datePicker.clear();
                Swal.fire({
                    icon: 'info',
                    title: 'Select Doctor First',
                    text: 'Please select a doctor first to see their available dates.',
                    timer: 3000,
                    showConfirmButton: false
                });
            }
        });

        function loadTimeSlots(date) {
            $('#time-slots-container').html(`
                <div class="text-center py-8">
                    <div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
                    <p class="mt-2 text-gray-600">Loading available slots...</p>
                </div>
            `);
            
            $.ajax({
                url: '<?php echo base_url('frontend/get_available_slots'); ?>',
                method: 'POST',
                data: { 
                    doctor_id: selectedDoctor,
                    date: date
                },
                dataType: 'json',
                success: function(slots) {
                    console.log('Time slots received:', slots);
                    let html = '';
                    
                    if (slots.length > 0) {
                        html = '<div class="grid grid-cols-2 md:grid-cols-4 gap-3">';
                        slots.forEach(function(slot) {
                            console.log('Rendering slot:', slot);
                            html += `
                                <div class="time-slot bg-white border-2 border-gray-200 rounded-lg px-4 py-3 text-center cursor-pointer hover:border-primary hover:bg-blue-50 transition-all duration-300"
                                     data-slot="${slot.slot}">
                                    <div class="text-sm font-medium text-gray-800">${slot.slot}</div>
                                </div>
                            `;
                        });
                        html += '</div>';
                    } else {
                        html = `
                            <div class="text-center py-8">
                                <div class="w-16 h-16 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-4">
                                    <i class="fas fa-calendar-times text-2xl text-gray-400"></i>
                                </div>
                                <p class="text-gray-500">No available slots for this date. Please select another date.</p>
                            </div>
                        `;
                    }
                    
                    $('#time-slots-container').html(html);
                    
                    $('.time-slot').on('click', function() {
                        $('.time-slot').removeClass('selected border-primary bg-primary text-white').addClass('border-gray-200 bg-white');
                        $(this).addClass('selected border-primary bg-primary text-white').removeClass('border-gray-200 bg-white');
                        selectedTimeSlot = $(this).data('slot');
                        // Only enable if visit type is also selected
                        if (selectedVisitType) {
                            $('#continueToPaymentBtn').prop('disabled', false);
                        }
                    });
                }
            });
        }

    </script>

    <!-- Include OTP Verification Modal -->
    <?php $this->load->view('includes/otp_verification_modal'); ?>
    
    <!-- Appointment OTP Modal (Pure Tailwind) -->
    <div id="appointmentOtpModal" class="fixed inset-0 z-50 hidden" aria-labelledby="appointmentOtpModalLabel" role="dialog" aria-modal="true">
        <!-- Background overlay -->
        <div class="fixed inset-0 bg-gray-500 bg-opacity-75 modal-overlay" onclick="closeAppointmentOtpModal()"></div>
        
        <!-- Modal container - perfectly centered -->
        <div class="fixed inset-0 flex items-center justify-center p-4">
            <!-- Modal panel -->
            <div class="bg-white rounded-lg shadow-xl max-w-lg w-full max-h-screen overflow-y-auto">
                <!-- Modal Header -->
                <div class="bg-gradient-to-r from-green-600 to-blue-600 px-6 py-4">
                    <div class="flex items-center justify-between">
                        <h3 class="text-lg font-medium text-white" id="appointmentOtpModalLabel">
                            <i class="fas fa-calendar-check mr-2"></i>Appointment Phone Verification
                        </h3>
                        <button type="button" class="text-white hover:text-gray-200 focus:outline-none" onclick="closeAppointmentOtpModal()">
                            <span class="sr-only">Close</span>
                            <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                            </svg>
                        </button>
                    </div>
                </div>
                
                <!-- Modal Body -->
                <div class="px-6 py-6">
                    <!-- Step 1: Send OTP -->
                    <div id="appointment_otp_step_1" class="text-center">
                        <div class="w-20 h-20 bg-gradient-to-r from-green-500 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-mobile-alt text-3xl text-white"></i>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800 mb-3">Verify Your Phone Number</h4>
                        <p class="text-gray-600 mb-3">We need to verify your phone number to continue with appointment booking.</p>
                        <p class="text-lg font-semibold text-green-600 mb-6" id="appointment_phone_display"></p>
                        
                        <button type="button" class="bg-green-600 hover:bg-green-700 text-white font-medium py-3 px-8 rounded-lg transition duration-200" id="send_appointment_otp_btn">
                            <i class="fas fa-paper-plane mr-2"></i>Send Verification Code
                        </button>
                    </div>

                    <!-- Step 2: Enter OTP -->
                    <div id="appointment_otp_step_2" class="text-center hidden">
                        <div class="w-20 h-20 bg-gradient-to-r from-green-500 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-sms text-3xl text-white"></i>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800 mb-2">Verification Code Sent</h4>
                        <p class="text-gray-600 mb-1">Enter the 6-digit code sent to</p>
                        <p class="text-lg font-semibold text-green-600 mb-6" id="appointment_otp_phone_display"></p>

                        <!-- OTP Input -->
                        <form id="appointmentOtpVerificationForm">
                            <div class="flex justify-center gap-2 mb-4">
                                <input type="text" class="appointment-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-green-500 focus:outline-none transition duration-200" maxlength="1" id="appointment_otp1" data-index="1" tabindex="1">
                                <input type="text" class="appointment-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-green-500 focus:outline-none transition duration-200" maxlength="1" id="appointment_otp2" data-index="2" tabindex="2">
                                <input type="text" class="appointment-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-green-500 focus:outline-none transition duration-200" maxlength="1" id="appointment_otp3" data-index="3" tabindex="3">
                                <input type="text" class="appointment-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-green-500 focus:outline-none transition duration-200" maxlength="1" id="appointment_otp4" data-index="4" tabindex="4">
                                <input type="text" class="appointment-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-green-500 focus:outline-none transition duration-200" maxlength="1" id="appointment_otp5" data-index="5" tabindex="5">
                                <input type="text" class="appointment-otp-input w-12 h-12 text-center text-lg font-semibold border-2 border-gray-300 rounded-lg focus:border-green-500 focus:outline-none transition duration-200" maxlength="1" id="appointment_otp6" data-index="6" tabindex="6">
                            </div>
                            <p class="text-sm text-gray-500 mb-4">Enter the 6-digit verification code</p>

                            <!-- Timer and Resend -->
                            <div class="mb-4">
                                <div id="appointment_otp_timer_container" class="hidden">
                                    <p class="text-gray-600 mb-2">
                                        <i class="fas fa-clock mr-1"></i>
                                        Code expires in: <span id="appointment_otp_timer" class="font-semibold text-yellow-600">05:00</span>
                                    </p>
                                </div>
                                <div id="appointment_otp_resend_container" class="hidden">
                                    <p class="text-gray-600 mb-2">Didn't receive the code?</p>
                                    <button type="button" class="text-green-600 hover:text-green-800 font-semibold focus:outline-none" id="appointment_resend_otp_btn">
                                        <i class="fas fa-redo mr-1"></i>Resend Code
                                    </button>
                                </div>
                            </div>

                            <!-- Message Area -->
                            <div id="appointment_otp_message" class="hidden p-3 rounded-lg mb-4"></div>

                            <!-- Action Buttons -->
                            <div class="flex justify-center gap-4">
                                <button type="button" class="bg-gray-500 hover:bg-gray-600 text-white font-medium py-2 px-6 rounded-lg transition duration-200" onclick="showAppointmentOtpStep1()">
                                    <i class="fas fa-arrow-left mr-2"></i>Back
                                </button>
                                <button type="submit" class="bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-6 rounded-lg transition duration-200" id="verify_appointment_otp_btn">
                                    <i class="fas fa-check mr-2"></i>Verify Code
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Appointment OTP Modal Variables
        let currentAppointmentPhone = '';
        let currentAppointmentPurpose = '';
        let appointmentOtpTimer = null;
        let appointmentOtpTimeLeft = 300; // 5 minutes

        // Show Appointment OTP Modal (Step 1)
        function showAppointmentOtpModal(phone, purpose) {
            currentAppointmentPhone = phone;
            currentAppointmentPurpose = purpose;
            
            $('#appointment_phone_display').text(phone);
            $('#appointment_otp_phone_display').text(phone);
            
            // Show step 1, hide step 2
            showAppointmentOtpStep1();
            
            // Show modal with Tailwind classes
            $('#appointmentOtpModal').removeClass('hidden');
            $('body').addClass('overflow-hidden'); // Prevent background scrolling
        }

        // Close Appointment OTP Modal
        function closeAppointmentOtpModal() {
            $('#appointmentOtpModal').addClass('hidden');
            $('body').removeClass('overflow-hidden'); // Restore background scrolling
            
            // Clear timer if running
            if (appointmentOtpTimer) {
                clearInterval(appointmentOtpTimer);
                appointmentOtpTimer = null;
            }
            
            // Reset form
            resetAppointmentOtpForm();
        }

        // Show Step 1 (Send OTP button)
        function showAppointmentOtpStep1() {
            $('#appointment_otp_step_1').removeClass('hidden');
            $('#appointment_otp_step_2').addClass('hidden');
            $('#appointmentOtpModalLabel').html('<i class="fas fa-calendar-check mr-2"></i>Appointment Phone Verification');
        }

        // Show Step 2 (OTP input fields)
        function showAppointmentOtpStep2() {
            $('#appointment_otp_step_1').addClass('hidden');
            $('#appointment_otp_step_2').removeClass('hidden');
            $('#appointmentOtpModalLabel').html('<i class="fas fa-sms mr-2"></i>Enter Verification Code');
            
            // Focus on first OTP input
            setTimeout(() => {
                $('#appointment_otp1').focus();
            }, 100);
            
            // Start timer
            startAppointmentOtpTimer();
        }

        // Reset OTP form
        function resetAppointmentOtpForm() {
            // Clear OTP inputs
            for (let i = 1; i <= 6; i++) {
                $('#appointment_otp' + i).val('');
            }
            
            // Hide message
            $('#appointment_otp_message').addClass('hidden');
            
            // Reset timer containers
            $('#appointment_otp_timer_container').addClass('hidden');
            $('#appointment_otp_resend_container').addClass('hidden');
            
            // Reset button states
            $('#send_appointment_otp_btn').prop('disabled', false).html('<i class="fas fa-paper-plane mr-2"></i>Send Verification Code');
            $('#verify_appointment_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-2"></i>Verify Code');
        }

        // Send OTP (triggered by button click in step 1)
        function sendAppointmentOtp() {
            $('#send_appointment_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Sending...');
            
            $.ajax({
                url: '<?php echo base_url('frontend/send_otp'); ?>',
                method: 'POST',
                data: {
                    phone: currentAppointmentPhone,
                    purpose: 'appointment_booking'
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        showAppointmentOtpStep2();
                    } else {
                        showAppointmentOtpMessage(response.message || 'Failed to send verification code', 'danger');
                    }
                },
                error: function() {
                    showAppointmentOtpMessage('Network error. Please try again.', 'danger');
                },
                complete: function() {
                    $('#send_appointment_otp_btn').prop('disabled', false).html('<i class="fas fa-paper-plane mr-2"></i>Send Verification Code');
                }
            });
        }

        // Timer functions
        function startAppointmentOtpTimer() {
            appointmentOtpTimeLeft = 300; // 5 minutes
            $('#appointment_otp_timer_container').removeClass('hidden');
            $('#appointment_otp_resend_container').addClass('hidden');
            
            appointmentOtpTimer = setInterval(function() {
                appointmentOtpTimeLeft--;
                updateAppointmentOtpTimer();
                
                if (appointmentOtpTimeLeft <= 0) {
                    clearInterval(appointmentOtpTimer);
                    $('#appointment_otp_timer_container').addClass('hidden');
                    $('#appointment_otp_resend_container').removeClass('hidden');
                }
            }, 1000);
        }

        function updateAppointmentOtpTimer() {
            let minutes = Math.floor(appointmentOtpTimeLeft / 60);
            let seconds = appointmentOtpTimeLeft % 60;
            $('#appointment_otp_timer').text(
                (minutes < 10 ? '0' : '') + minutes + ':' + 
                (seconds < 10 ? '0' : '') + seconds
            );
        }

        // OTP input handling
        function setupAppointmentOtpInputs() {
            // Handle input navigation for appointment OTP inputs using class
            $('.appointment-otp-input').on('input', function() {
                if (this.value.length === 1) {
                    let currentIndex = parseInt($(this).data('index'));
                    if (currentIndex < 6) {
                        $('#appointment_otp' + (currentIndex + 1)).focus();
                    }
                }
            });

            $('.appointment-otp-input').on('keydown', function(e) {
                if (e.key === 'Backspace' && this.value.length === 0) {
                    let currentIndex = parseInt($(this).data('index'));
                    if (currentIndex > 1) {
                        $('#appointment_otp' + (currentIndex - 1)).focus();
                    }
                }
            });

            $('.appointment-otp-input').on('paste', function(e) {
                e.preventDefault();
                let paste = (e.originalEvent.clipboardData || window.clipboardData).getData('text');
                if (paste.length === 6 && /^\d+$/.test(paste)) {
                    $('.appointment-otp-input').each(function(index) {
                        $(this).val(paste.charAt(index));
                    });
                    $('#appointment_otp6').focus();
                }
            });
        }

        // Verify OTP
        function verifyAppointmentOtp() {
            let otp = '';
            
            // Collect OTP from inputs using class selector
            $('.appointment-otp-input').each(function() {
                let value = $(this).val() || '';
                otp += value;
            });
            
            if (otp.length !== 6 || !/^\d{6}$/.test(otp)) {
                showAppointmentOtpMessage('Please enter the complete 6-digit code', 'danger');
                return;
            }
            
            $('#verify_appointment_otp_btn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Verifying...');
            
            $.ajax({
                url: '<?php echo base_url('frontend/verify_otp'); ?>',
                method: 'POST',
                data: {
                    phone: currentAppointmentPhone,
                    otp_code: otp,
                    purpose: 'appointment_booking'
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        showAppointmentOtpMessage('✅ Phone verified successfully!', 'success');
                        setTimeout(() => {
                            closeAppointmentOtpModal();
                            handleAppointmentOtpSuccess();
                        }, 1500);
                    } else {
                        showAppointmentOtpMessage(response.message || 'Invalid verification code', 'danger');
                    }
                },
                error: function() {
                    showAppointmentOtpMessage('Network error. Please try again.', 'danger');
                },
                complete: function() {
                    $('#verify_appointment_otp_btn').prop('disabled', false).html('<i class="fas fa-check mr-2"></i>Verify Code');
                }
            });
        }

        // Handle successful OTP verification
        function handleAppointmentOtpSuccess() {
            onPhoneVerifiedForAppointment(currentAppointmentPhone);
        }

        // Show message in OTP modal
        function showAppointmentOtpMessage(message, type) {
            let bgColor = '';
            let textColor = '';
            let borderColor = '';
            
            if (type === 'success') {
                bgColor = 'bg-green-100';
                textColor = 'text-green-800';
                borderColor = 'border-green-400';
            } else if (type === 'danger' || type === 'error') {
                bgColor = 'bg-red-100';
                textColor = 'text-red-800';
                borderColor = 'border-red-400';
            } else if (type === 'warning') {
                bgColor = 'bg-yellow-100';
                textColor = 'text-yellow-800';
                borderColor = 'border-yellow-400';
            } else {
                bgColor = 'bg-blue-100';
                textColor = 'text-blue-800';
                borderColor = 'border-blue-400';
            }
            
            $('#appointment_otp_message')
                .removeClass()
                .addClass(`${bgColor} ${textColor} ${borderColor} border p-3 rounded-lg mb-4`)
                .text(message)
                .removeClass('hidden');
                
            setTimeout(() => {
                $('#appointment_otp_message').addClass('hidden');
            }, 5000);
        }

        // Initialize Appointment OTP Modal
        $(document).ready(function() {
            setupAppointmentOtpInputs();
            
            // Send OTP button click
            $('#send_appointment_otp_btn').click(function() {
                sendAppointmentOtp();
            });
            
            // Resend OTP button click
            $('#appointment_resend_otp_btn').click(function() {
                sendAppointmentOtp();
            });
            
            // OTP form submission
            $('#appointmentOtpVerificationForm').submit(function(e) {
                e.preventDefault();
                verifyAppointmentOtp();
            });
            
            // Keyboard event handling
            $(document).keydown(function(e) {
                if (e.key === 'Escape' && !$('#appointmentOtpModal').hasClass('hidden')) {
                    closeAppointmentOtpModal();
                }
            });
        });

        // OTP verification success callback
        window.otpVerificationSuccess = function(phone, purpose) {
            if (purpose === 'appointment_booking') {
                // Phone verified for appointment booking
                onPhoneVerifiedForAppointment(phone);
            }
        };
    </script>
</body>
</html> 