<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Queue_model extends CI_Model
{
    function __construct()
    {
        parent::__construct();
    }

    // Generate queue numbers for today's confirmed appointments
    public function generateTodayQueue($doctor_id, $date = null)
    {
        // Force today's date for queue generation - ignore any passed date
        $date = date('Y-m-d');

        // Enhanced debugging
        log_message('debug', '=== generateTodayQueue START ===');
        log_message('debug', 'Doctor ID: ' . $doctor_id . ', Date: ' . $date);
        
        $session_hospital_id = $this->session->userdata('hospital_id');
        log_message('debug', 'Session Hospital ID: ' . ($session_hospital_id ?: 'NULL'));

        // Use EXACT timestamp matching like Today's Appointment page
        $today_timestamp = strtotime($date);
        log_message('debug', 'Today timestamp: ' . $today_timestamp . ' (' . date('Y-m-d H:i:s', $today_timestamp) . ')');

        // First, let's check if ANY appointments exist for today for this doctor
        $this->db->where('doctor', $doctor_id);
        $this->db->where('date', $today_timestamp);
        $total_appointments = $this->db->count_all_results('appointment');
        log_message('debug', 'Total appointments for doctor ' . $doctor_id . ' on ' . $date . ': ' . $total_appointments);

        // Reset query builder
        $this->db->reset_query();

        // Check confirmed appointments
        $this->db->where('doctor', $doctor_id);
        $this->db->where('date', $today_timestamp);
        $this->db->where('status IN ("Confirmed", "Pending Confirmation")');
        $confirmed_appointments = $this->db->count_all_results('appointment');
        log_message('debug', 'Confirmed appointments for doctor ' . $doctor_id . ' on ' . $date . ': ' . $confirmed_appointments);

        // Reset query builder
        $this->db->reset_query();

        // Get confirmed appointments for the doctor on the specified date that don't have queue numbers
        $this->db->where('doctor', $doctor_id);
        $this->db->where('date', $today_timestamp);
        $this->db->where('status IN ("Confirmed", "Pending Confirmation")');
        $this->db->where('(queue_number IS NULL OR queue_number = 0)');
        
        // Only filter by hospital_id if session hospital_id is set and not empty
        if (!empty($session_hospital_id) && $session_hospital_id !== 'superadmin') {
            $this->db->where('hospital_id', $session_hospital_id);
            log_message('debug', 'Applying hospital filter: ' . $session_hospital_id);
        } else {
            log_message('debug', 'No hospital filter applied');
        }
        
        $this->db->order_by('s_time', 'ASC');
        $appointments = $this->db->get('appointment')->result();
        
        // Debug logging
        log_message('debug', 'generateTodayQueue SQL: ' . $this->db->last_query());
        log_message('debug', 'generateTodayQueue found appointments: ' . count($appointments));
        
        if (count($appointments) > 0) {
            log_message('debug', 'Appointments found: ' . json_encode(array_map(function($apt) {
                return ['id' => $apt->id, 'patient' => $apt->patient, 'status' => $apt->status, 'hospital_id' => $apt->hospital_id];
            }, $appointments)));
        }

        // Get the highest existing queue number for the day
        $this->db->select_max('queue_number');
        $this->db->where('doctor', $doctor_id);
        $this->db->where('date', $today_timestamp);
        // Only filter by hospital_id if session hospital_id is set and not empty
        if (!empty($session_hospital_id) && $session_hospital_id !== 'superadmin') {
            $this->db->where('hospital_id', $session_hospital_id);
        }
        $max_queue = $this->db->get('appointment')->row();
        
        $next_queue_number = ($max_queue->queue_number ?? 0) + 1;

        // Assign queue numbers
        foreach ($appointments as $appointment) {
            $queue_data = [
                'queue_number' => $next_queue_number,
                'queue_status' => 'waiting',
                'queue_date' => $date
            ];
            
            $this->db->where('id', $appointment->id);
            $update_result = $this->db->update('appointment', $queue_data);
            
            log_message('debug', 'Updating appointment ' . $appointment->id . ' with queue number ' . $next_queue_number . ': ' . ($update_result ? 'SUCCESS' : 'FAILED'));
            log_message('debug', 'Update SQL: ' . $this->db->last_query());

            // Log the queue assignment (with error handling)
            try {
                $this->logQueueAction($appointment->id, $doctor_id, $appointment->patient, $next_queue_number, 'queued', null, 'waiting');
            } catch (Exception $e) {
                log_message('debug', 'logQueueAction failed: ' . $e->getMessage() . ' - Continuing anyway');
            }
            
            $next_queue_number++;
        }

        log_message('debug', '=== generateTodayQueue END - Processed ' . count($appointments) . ' appointments ===');
        return count($appointments);
    }

    // Get today's queue for a specific doctor
    public function getTodayQueue($doctor_id, $date = null)
    {
        // Force today's date for queue page - ignore any passed date
        $date = date('Y-m-d');
        log_message('debug', 'getTodayQueue: Using date = ' . $date . ' (PHP timezone: ' . date_default_timezone_get() . ')');

        $this->db->select('
            a.id as appointment_id,
            a.queue_number,
            a.queue_status,
            a.date,
            a.s_time as time,
            a.treated_at,
            a.remarks as problem,
            a.status as appointment_status,
            p.id as patient_id,
            p.name as patient_name,
            p.phone as patient_phone,
            p.age as patient_age,
            d.name as doctor_name,
            h.name as hospital_name,
            a.hospital_id,
            p.sex as patient_gender
        ');
        $this->db->from('appointment a');
        $this->db->join('patient p', 'a.patient = p.id', 'left');
        $this->db->join('doctor d', 'a.doctor = d.id', 'left');
        $this->db->join('hospital h', 'a.hospital_id = h.id', 'left');
        
        // Use EXACT timestamp matching like Today's Appointment page
        $today_timestamp = strtotime($date);
        log_message('debug', 'getTodayQueue using timestamp: ' . $today_timestamp . ' (' . date('Y-m-d H:i:s', $today_timestamp) . ')');
        
        $this->db->where('a.doctor', $doctor_id);
        $this->db->where('a.date', $today_timestamp);
        $this->db->where('a.status IN ("Confirmed", "Pending Confirmation", "Treated")');
        $this->db->where('a.queue_number IS NOT NULL');
        $this->db->where('a.queue_number > 0');
        // Only filter by hospital_id if session hospital_id is set and not empty
        $session_hospital_id = $this->session->userdata('hospital_id');
        if (!empty($session_hospital_id) && $session_hospital_id !== 'superadmin') {
            $this->db->where('a.hospital_id', $session_hospital_id);
        }
        $this->db->order_by('a.queue_number', 'ASC');
        
        $result = $this->db->get()->result();
        
        // Debug logging - remove after fixing
        log_message('debug', 'getTodayQueue SQL: ' . $this->db->last_query());
        log_message('debug', 'getTodayQueue result count: ' . count($result));
        log_message('debug', 'Doctor ID: ' . $doctor_id . ', Date: ' . $date . ', Session Hospital ID: ' . $session_hospital_id);
        
        // TEMPORARY DEBUG - Show what we're getting
        if (!empty($result)) {
            foreach ($result as $item) {
                log_message('debug', 'Queue item: ID=' . $item->appointment_id . ', Date=' . date('Y-m-d H:i:s', $item->date) . ', Queue#=' . $item->queue_number);
            }
        } else {
            log_message('debug', 'NO QUEUE ITEMS FOUND - checking why...');
            // Check if appointments exist for today at all
            $this->db->select('COUNT(*) as total');
            $this->db->from('appointment');
            $this->db->where('doctor', $doctor_id);
            $this->db->where('date', $today_timestamp);
            $count_result = $this->db->get()->row();
            log_message('debug', 'Total appointments for doctor ' . $doctor_id . ' on ' . $date . ': ' . $count_result->total);
        }
        
        return $result;
    }

    // Get current patient being treated
    public function getCurrentPatient($doctor_id, $date = null)
    {
        // Force today's date for queue page - ignore any passed date
        $date = date('Y-m-d');
        $today_timestamp = strtotime($date);

        $this->db->select('
            a.id as appointment_id,
            a.queue_number,
            a.queue_status,
            a.s_time as time,
            p.name as patient_name,
            p.phone as patient_phone
        ');
        $this->db->from('appointment a');
        $this->db->join('patient p', 'a.patient = p.id', 'left');
        $this->db->where('a.doctor', $doctor_id);
        $this->db->where('a.date', $today_timestamp);
        $this->db->where('a.queue_status', 'in_progress');
        // Only filter by hospital_id if session hospital_id is set and not empty
        $session_hospital_id = $this->session->userdata('hospital_id');
        if (!empty($session_hospital_id) && $session_hospital_id !== 'superadmin') {
            $this->db->where('a.hospital_id', $session_hospital_id);
        }
        
        return $this->db->get()->row();
    }

    // Get next patient in queue
    public function getNextPatient($doctor_id, $date = null)
    {
        // Force today's date for queue page - ignore any passed date
        $date = date('Y-m-d');
        $today_timestamp = strtotime($date);

        $this->db->select('
            a.id as appointment_id,
            a.queue_number,
            a.s_time as time,
            p.name as patient_name,
            p.phone as patient_phone
        ');
        $this->db->from('appointment a');
        $this->db->join('patient p', 'a.patient = p.id', 'left');
        $this->db->where('a.doctor', $doctor_id);
        $this->db->where('a.date', $today_timestamp);
        $this->db->where('a.queue_status', 'waiting');
        // Only filter by hospital_id if session hospital_id is set and not empty
        $session_hospital_id = $this->session->userdata('hospital_id');
        if (!empty($session_hospital_id) && $session_hospital_id !== 'superadmin') {
            $this->db->where('a.hospital_id', $session_hospital_id);
        }
        $this->db->order_by('a.queue_number', 'ASC');
        $this->db->limit(1);
        
        return $this->db->get()->row();
    }

    // Mark patient as treated
    public function markPatientTreated($appointment_id, $action_by)
    {
        // Get current appointment details
        $appointment = $this->db->get_where('appointment', ['id' => $appointment_id])->row();
        if (!$appointment) {
            return false;
        }

        // Update appointment status
        $update_data = [
            'queue_status' => 'treated',
            'status' => 'Treated',
            'treated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $appointment_id);
        $result = $this->db->update('appointment', $update_data);

        if ($result) {
            // Log the action
            $this->logQueueAction(
                $appointment_id, 
                $appointment->doctor, 
                $appointment->patient, 
                $appointment->queue_number, 
                'treated', 
                $appointment->queue_status, 
                'treated',
                $action_by
            );
        }

        return $result;
    }

    // Start treating a patient (mark as in_progress)
    public function startTreatingPatient($appointment_id, $action_by)
    {
        // Get current appointment details
        $appointment = $this->db->get_where('appointment', ['id' => $appointment_id])->row();
        if (!$appointment) {
            return false;
        }

        // Update queue status
        $update_data = [
            'queue_status' => 'in_progress'
        ];
        
        $this->db->where('id', $appointment_id);
        $result = $this->db->update('appointment', $update_data);

        if ($result) {
            // Log the action
            $this->logQueueAction(
                $appointment_id, 
                $appointment->doctor, 
                $appointment->patient, 
                $appointment->queue_number, 
                'started', 
                $appointment->queue_status, 
                'in_progress',
                $action_by
            );
        }

        return $result;
    }

    // Get patient's position in queue
    public function getPatientQueuePosition($patient_id, $doctor_id, $date = null)
    {
        if ($date === null) {
            $date = date('Y-m-d');
        }

        $this->db->select('
            a.id as appointment_id,
            a.queue_number,
            a.queue_status,
            a.s_time as time,
            d.name as doctor_name,
            COUNT(a2.id) as position_in_queue
        ');
        $this->db->from('appointment a');
        $this->db->join('doctor d', 'a.doctor = d.id', 'left');
        $this->db->join('appointment a2', 'a2.doctor = a.doctor AND a2.queue_number < a.queue_number AND a2.queue_status IN ("waiting", "in_progress") AND DATE(FROM_UNIXTIME(a2.date)) = DATE(FROM_UNIXTIME(a.date))', 'left');
        $this->db->where('a.patient', $patient_id);
        $this->db->where('a.doctor', $doctor_id);
        $this->db->where('DATE(FROM_UNIXTIME(a.date))', $date);
        $this->db->where('a.queue_number IS NOT NULL');
        // Only filter by hospital_id if session hospital_id is set and not empty
        $session_hospital_id = $this->session->userdata('hospital_id');
        if (!empty($session_hospital_id) && $session_hospital_id !== 'superadmin') {
            $this->db->where('a.hospital_id', $session_hospital_id);
        }
        $this->db->group_by('a.id');
        
        return $this->db->get()->row();
    }

    // Get queue statistics for doctor dashboard
    public function getQueueStats($doctor_id, $date = null)
    {
        // Force today's date for queue page - ignore any passed date
        $date = date('Y-m-d');
        $today_timestamp = strtotime($date);

        $stats = [
            'total_patients' => 0,
            'waiting' => 0,
            'in_progress' => 0,
            'treated' => 0,
            'missed' => 0
        ];

        $this->db->select('queue_status, COUNT(*) as count');
        $this->db->from('appointment');
        $this->db->where('doctor', $doctor_id);
        $this->db->where('date', $today_timestamp);
        $this->db->where('queue_number IS NOT NULL');
        // Only filter by hospital_id if session hospital_id is set and not empty
        $session_hospital_id = $this->session->userdata('hospital_id');
        if (!empty($session_hospital_id) && $session_hospital_id !== 'superadmin') {
            $this->db->where('hospital_id', $session_hospital_id);
        }
        $this->db->group_by('queue_status');
        
        $results = $this->db->get()->result();
        
        foreach ($results as $result) {
            $stats[$result->queue_status] = $result->count;
            $stats['total_patients'] += $result->count;
        }

        return $stats;
    }

    // Log queue actions
    private function logQueueAction($appointment_id, $doctor_id, $patient_id, $queue_number, $action, $previous_status, $new_status, $action_by = null, $notes = null)
    {
        if ($action_by === null) {
            $action_by = $this->ion_auth->get_user_id();
        }

        $log_data = [
            'appointment_id' => $appointment_id,
            'doctor_id' => $doctor_id,
            'patient_id' => $patient_id,
            'queue_number' => $queue_number,
            'action' => $action,
            'previous_status' => $previous_status,
            'new_status' => $new_status,
            'action_by' => $action_by,
            'notes' => $notes,
            'hospital_id' => $this->session->userdata('hospital_id') ?: 1
        ];

        return $this->db->insert('appointment_queue_log', $log_data);
    }

    // Get queue history/log
    public function getQueueLog($doctor_id, $date = null, $limit = 50)
    {
        if ($date === null) {
            $date = date('Y-m-d');
        }

        $this->db->select('
            ql.*,
            p.name as patient_name,
            u.username as action_by_name
        ');
        $this->db->from('appointment_queue_log ql');
        $this->db->join('patient p', 'ql.patient_id = p.id');
        $this->db->join('users u', 'ql.action_by = u.id');
        $this->db->where('ql.doctor_id', $doctor_id);
        $this->db->where('DATE(ql.action_time)', $date);
        // Only filter by hospital_id if session hospital_id is set and not empty
        $session_hospital_id = $this->session->userdata('hospital_id');
        if (!empty($session_hospital_id) && $session_hospital_id !== 'superadmin') {
            $this->db->where('ql.hospital_id', $session_hospital_id);
        }
        $this->db->order_by('ql.action_time', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result();
    }
} 