<?php
if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Sms_model extends CI_Model {

    function __construct() {
        parent::__construct();
    }

    /**
     * Get SMS template by type, language and hospital
     */
    public function getSmsTemplate($type, $hospital_id, $language = 'english') {
        $this->db->where('type', $type);
        $this->db->where('hospital_id', $hospital_id);
        $this->db->where('language', $language);
        $this->db->where('status', 'Active');
        $query = $this->db->get('autosmstemplate');
        
        if ($query->num_rows() > 0) {
            return $query->row();
        }
        
        // Fallback to superadmin template if hospital-specific not found
        $this->db->where('type', $type);
        $this->db->where('hospital_id', 'superadmin');
        $this->db->where('language', $language);
        $this->db->where('status', 'Active');
        $query = $this->db->get('autosmstemplate');
        
        return $query->num_rows() > 0 ? $query->row() : null;
    }

    /**
     * Check if SMS permission is enabled for hospital
     */
    public function checkSmsPermission($hospital_id, $permission_type) {
        // Get hospital settings from hospital table
        $this->db->where('id', $hospital_id);
        $hospital = $this->db->get('hospital')->row();
        
        if (!$hospital) return false;

        // Check specific SMS permission based on type using existing permission fields
        switch ($permission_type) {
            case 'patient_registration':
            case 'patient_update':
                // Use existing patient SMS permission
                return isset($hospital->patient_sms) && $hospital->patient_sms == 'yes';
            
            case 'doctor_registration':
            case 'doctor_update':
                // Use existing doctor SMS permission
                return isset($hospital->doctor_sms) && $hospital->doctor_sms == 'yes';
            
            case 'online_appointment_booking':
            case 'appointment_without_payment':
            case 'ivr_appointment_booking':
                // Use existing appointment SMS permission
                return isset($hospital->appointment_sms) && $hospital->appointment_sms == 'yes';
            
            default:
                return false;
        }
    }

    /**
     * Check doctor SMS permission using existing doctor table structure
     */
    public function checkDoctorSmsPermission($doctor_id, $permission_type) {
        $this->db->where('id', $doctor_id);
        $doctor = $this->db->get('doctor')->row();
        
        if (!$doctor) return false;

        // Use existing SMS permission fields in doctor table
        switch ($permission_type) {
            case 'doctor_registration':
            case 'doctor_update':
                // Check if doctor has SMS enabled (existing field)
                return isset($doctor->sms) && $doctor->sms == 'yes';
            
            default:
                return false;
        }
    }

    /**
     * Replace shortcodes in SMS message
     */
    public function replaceShortcodes($message, $data) {
        foreach ($data as $key => $value) {
            $message = str_replace('{' . $key . '}', $value, $message);
        }
        return $message;
    }

    /**
     * Send patient registration SMS
     */
    public function sendPatientRegistrationSms($patient_data, $hospital_id) {
        // Check permission
        if (!$this->checkSmsPermission($hospital_id, 'patient_registration')) {
            return false;
        }

        // Get hospital language setting
        $language = $this->getHospitalLanguage($hospital_id);
        
        // Get SMS template
        $template = $this->getSmsTemplate('patient_registration', $hospital_id, $language);
        if (!$template) return false;

        // Prepare shortcode data
        $shortcode_data = [
            'name' => $patient_data['name'],
            'firstname' => $patient_data['firstname'] ?? '',
            'lastname' => $patient_data['lastname'] ?? '',
            'patient_id' => $patient_data['patient_id'],
            'phone' => $patient_data['phone'],
            'hospital_name' => $this->getHospitalName($hospital_id)
        ];

        // Replace shortcodes
        $message = $this->replaceShortcodes($template->message, $shortcode_data);

        // Send SMS
        return $this->sendSms($patient_data['phone'], $message, $hospital_id);
    }

    /**
     * Send patient update SMS
     */
    public function sendPatientUpdateSms($patient_data, $hospital_id) {
        if (!$this->checkSmsPermission($hospital_id, 'patient_update')) {
            return false;
        }

        $language = $this->getHospitalLanguage($hospital_id);
        $template = $this->getSmsTemplate('patient_update', $hospital_id, $language);
        if (!$template) return false;

        $shortcode_data = [
            'name' => $patient_data['name'],
            'firstname' => $patient_data['firstname'] ?? '',
            'lastname' => $patient_data['lastname'] ?? '',
            'patient_id' => $patient_data['patient_id'],
            'hospital_name' => $this->getHospitalName($hospital_id)
        ];

        $message = $this->replaceShortcodes($template->message, $shortcode_data);
        return $this->sendSms($patient_data['phone'], $message, $hospital_id);
    }

    /**
     * Send doctor registration SMS
     */
    public function sendDoctorRegistrationSms($doctor_data, $hospital_id) {
        // Check both hospital and doctor permissions using existing fields
        if (!$this->checkSmsPermission($hospital_id, 'doctor_registration') || 
            !$this->checkDoctorSmsPermission($doctor_data['id'], 'doctor_registration')) {
            return false;
        }

        $language = $this->getHospitalLanguage($hospital_id);
        $template = $this->getSmsTemplate('doctor_registration', $hospital_id, $language);
        if (!$template) return false;

        $shortcode_data = [
            'name' => $doctor_data['name'],
            'firstname' => $doctor_data['firstname'] ?? '',
            'lastname' => $doctor_data['lastname'] ?? '',
            'department' => $doctor_data['department'],
            'specialization' => $doctor_data['specialization'] ?? '',
            'hospital_name' => $this->getHospitalName($hospital_id)
        ];

        $message = $this->replaceShortcodes($template->message, $shortcode_data);
        return $this->sendSms($doctor_data['phone'], $message, $hospital_id);
    }

    /**
     * Send doctor update SMS
     */
    public function sendDoctorUpdateSms($doctor_data, $hospital_id) {
        if (!$this->checkSmsPermission($hospital_id, 'doctor_update') || 
            !$this->checkDoctorSmsPermission($doctor_data['id'], 'doctor_update')) {
            return false;
        }

        $language = $this->getHospitalLanguage($hospital_id);
        $template = $this->getSmsTemplate('doctor_update', $hospital_id, $language);
        if (!$template) return false;

        $shortcode_data = [
            'name' => $doctor_data['name'],
            'firstname' => $doctor_data['firstname'] ?? '',
            'lastname' => $doctor_data['lastname'] ?? '',
            'department' => $doctor_data['department'],
            'hospital_name' => $this->getHospitalName($hospital_id)
        ];

        $message = $this->replaceShortcodes($template->message, $shortcode_data);
        return $this->sendSms($doctor_data['phone'], $message, $hospital_id);
    }

    /**
     * Send online appointment booking SMS
     */
    public function sendOnlineAppointmentBookingSms($appointment_data, $hospital_id) {
        if (!$this->checkSmsPermission($hospital_id, 'online_appointment_booking')) {
            return false;
        }

        $language = $this->getHospitalLanguage($hospital_id);
        $template = $this->getSmsTemplate('online_appointment_booking', $hospital_id, $language);
        if (!$template) return false;

        $shortcode_data = [
            'name' => $appointment_data['patient_name'],
            'firstname' => $appointment_data['patient_firstname'] ?? '',
            'lastname' => $appointment_data['patient_lastname'] ?? '',
            'doctorname' => $appointment_data['doctor_name'],
            'appoinmentdate' => $appointment_data['appointment_date'],
            'time_slot' => $appointment_data['time_slot'],
            'hospital_name' => $this->getHospitalName($hospital_id)
        ];

        $message = $this->replaceShortcodes($template->message, $shortcode_data);
        return $this->sendSms($appointment_data['patient_phone'], $message, $hospital_id);
    }

    /**
     * Send appointment without payment SMS
     */
    public function sendAppointmentWithoutPaymentSms($appointment_data, $hospital_id) {
        if (!$this->checkSmsPermission($hospital_id, 'appointment_without_payment')) {
            return false;
        }

        $language = $this->getHospitalLanguage($hospital_id);
        $template = $this->getSmsTemplate('appointment_without_payment', $hospital_id, $language);
        if (!$template) return false;

        $shortcode_data = [
            'name' => $appointment_data['patient_name'],
            'firstname' => $appointment_data['patient_firstname'] ?? '',
            'lastname' => $appointment_data['patient_lastname'] ?? '',
            'doctorname' => $appointment_data['doctor_name'],
            'appoinmentdate' => $appointment_data['appointment_date'],
            'time_slot' => $appointment_data['time_slot'],
            'amount' => $appointment_data['amount'],
            'hospital_name' => $this->getHospitalName($hospital_id)
        ];

        $message = $this->replaceShortcodes($template->message, $shortcode_data);
        return $this->sendSms($appointment_data['patient_phone'], $message, $hospital_id);
    }

    /**
     * Send IVR appointment booking SMS
     */
    public function sendIvrAppointmentBookingSms($appointment_data, $hospital_id) {
        if (!$this->checkSmsPermission($hospital_id, 'ivr_appointment_booking')) {
            return false;
        }

        $language = $this->getHospitalLanguage($hospital_id);
        $template = $this->getSmsTemplate('ivr_appointment_booking', $hospital_id, $language);
        if (!$template) return false;

        $shortcode_data = [
            'name' => $appointment_data['patient_name'],
            'firstname' => $appointment_data['patient_firstname'] ?? '',
            'lastname' => $appointment_data['patient_lastname'] ?? '',
            'doctorname' => $appointment_data['doctor_name'],
            'appoinmentdate' => $appointment_data['appointment_date'],
            'time_slot' => $appointment_data['time_slot'],
            'hospital_name' => $this->getHospitalName($hospital_id)
        ];

        $message = $this->replaceShortcodes($template->message, $shortcode_data);
        return $this->sendSms($appointment_data['patient_phone'], $message, $hospital_id);
    }

    /**
     * Get hospital language setting
     */
    private function getHospitalLanguage($hospital_id) {
        $this->db->where('hospital_id', $hospital_id);
        $settings = $this->db->get('settings')->row();
        return $settings->language ?? 'english';
    }

    /**
     * Get hospital name
     */
    private function getHospitalName($hospital_id) {
        if ($hospital_id == 'superadmin') {
            $this->db->where('hospital_id', 'superadmin');
            $settings = $this->db->get('settings')->row();
            return $settings->system_vendor ?? 'Hospital Management System';
        }
        
        $this->db->where('id', $hospital_id);
        $hospital = $this->db->get('hospital')->row();
        return $hospital->name ?? 'Hospital';
    }

    /**
     * Send SMS using existing SMS gateway
     */
    private function sendSms($phone, $message, $hospital_id) {
        try {
            // Load existing SMS library/model if available
            if (class_exists('Sms')) {
                $sms = new Sms();
                return $sms->send($phone, $message, $hospital_id);
            }
            
            // Or use existing SMS sending method
            $CI =& get_instance();
            if (method_exists($CI, 'send_sms')) {
                return $CI->send_sms($phone, $message, $hospital_id);
            }
            
            // Log the SMS for debugging
            log_message('info', "AUTO SMS: Phone: $phone, Message: $message, Hospital: $hospital_id");
            
            // Return true for now (implement actual SMS gateway later)
            return true;
            
        } catch (Exception $e) {
            log_message('error', 'SMS sending failed: ' . $e->getMessage());
            return false;
        }
    }
}
?> 