<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Simple OTP Library - Basic phone verification without complex settings
 * Just send OTP and verify - that's it!
 */
class Simple_otp
{

    private $CI;

    public function __construct()
    {
        $this->CI = &get_instance();
        $this->CI->load->database();
        $this->CI->load->model('sms/sms_model');
        $this->CI->load->model('settings/settings_model');
    }

    /**
     * Send OTP to phone number
     * @param string $phone Phone number
     * @param string $purpose Purpose (patient_registration, etc)
     * @return array Result with success status and message
     */
    public function sendOTP($phone, $purpose = 'verification')
    {
        try {
            // Check if otp_verification table exists
            if (!$this->CI->db->table_exists('otp_verification')) {
                return array('success' => false, 'message' => 'OTP table does not exist. Please create the otp_verification table.');
            }

            // Clean phone number (remove non-digits except +)
            $phone = preg_replace('/[^\d+]/', '', $phone);

            // Basic validation - accept any number with 8+ digits
            if (strlen($phone) < 8) {
                return array('success' => false, 'message' => 'Invalid phone number - must be at least 8 digits');
            }

            // Check cooldown - prevent spam (60 seconds)
            if ($this->isInCooldown($phone, $purpose)) {
                return array('success' => false, 'message' => 'Please wait 60 seconds before requesting another OTP');
            }

            // Generate 6-digit OTP
            $otp_code = sprintf('%06d', mt_rand(0, 999999));

            // Expires in 5 minutes
            $expires_at = date('Y-m-d H:i:s', strtotime('+5 minutes'));

            // Store OTP in database
            $otp_data = array(
                'phone' => $phone,
                'otp_code' => $otp_code,
                'purpose' => $purpose,
                'expires_at' => $expires_at,
                'is_verified' => 0,
                'attempts' => 0,
                'hospital_id' => $this->CI->session->userdata('hospital_id') ?: 1,
                'created_at' => date('Y-m-d H:i:s')
            );

            $this->CI->db->insert('otp_verification', $otp_data);
            $otp_id = $this->CI->db->insert_id();

            if (!$otp_id) {
                return array('success' => false, 'message' => 'Failed to generate OTP');
            }

            // Send SMS using the same method as backend patient module
            $sms_result = $this->sendOTPSMSBackendStyle($phone, $otp_code, $purpose);

            if ($sms_result['success']) {
                return array(
                    'success' => true,
                    'message' => 'OTP sent successfully'
                );
            } else {
                // Delete OTP if SMS failed
                $this->CI->db->delete('otp_verification', array('id' => $otp_id));
                return array('success' => false, 'message' => 'Failed to send OTP: ' . $sms_result['message']);
            }
        } catch (Exception $e) {
            log_message('error', 'Simple OTP Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'System error: ' . $e->getMessage());
        }
    }

    /**
     * Verify OTP code
     * @param string $phone Phone number
     * @param string $otp_code OTP code entered by user
     * @param string $purpose Purpose
     * @return array Result
     */
    public function verifyOTP($phone, $otp_code, $purpose = 'verification')
    {
        try {
            // Clean inputs
            $phone = preg_replace('/[^\d+]/', '', $phone);
            $otp_code = preg_replace('/[^\d]/', '', $otp_code);

            if (empty($phone) || empty($otp_code)) {
                return array('success' => false, 'message' => 'Phone number and OTP code are required');
            }

            // Find active OTP
            $this->CI->db->where('phone', $phone);
            $this->CI->db->where('purpose', $purpose);
            $this->CI->db->where('is_verified', 0);
            $this->CI->db->where('expires_at >', date('Y-m-d H:i:s'));
            $this->CI->db->order_by('created_at', 'DESC');
            $otp_record = $this->CI->db->get('otp_verification')->row();

            if (!$otp_record) {
                return array('success' => false, 'message' => 'No valid OTP found. Please request a new one.');
            }

            // Check attempts (max 3)
            if ($otp_record->attempts >= 3) {
                return array('success' => false, 'message' => 'Too many failed attempts. Please request a new OTP.');
            }

            // Increment attempts
            $this->CI->db->where('id', $otp_record->id);
            $this->CI->db->update('otp_verification', array('attempts' => $otp_record->attempts + 1));

            // Verify OTP
            if ($otp_record->otp_code === $otp_code) {
                // Mark as verified
                $this->CI->db->where('id', $otp_record->id);
                $this->CI->db->update('otp_verification', array(
                    'is_verified' => 1,
                    'verified_at' => date('Y-m-d H:i:s')
                ));

                return array('success' => true, 'message' => 'OTP verified successfully');
            } else {
                return array('success' => false, 'message' => 'Invalid OTP code');
            }
        } catch (Exception $e) {
            log_message('error', 'OTP Verify Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'System error occurred');
        }
    }

    /**
     * Check if phone number was recently verified (for registration purposes)
     * @param string $phone Phone number
     * @param string $purpose Purpose (patient_registration, etc)
     * @param int $minutes_valid How many minutes the verification is valid (default 30)
     * @return bool True if verified within the time window
     */
    public function hasVerifiedOTP($phone, $purpose = 'verification', $minutes_valid = 30)
    {
        try {
            // Clean phone number
            $phone = preg_replace('/[^\d+]/', '', $phone);

            if (empty($phone)) {
                return false;
            }

            // Find recent verified OTP
            $time_threshold = date('Y-m-d H:i:s', strtotime("-{$minutes_valid} minutes"));

            $this->CI->db->where('phone', $phone);
            $this->CI->db->where('purpose', $purpose);
            $this->CI->db->where('is_verified', 1);
            $this->CI->db->where('verified_at >', $time_threshold);
            $this->CI->db->order_by('verified_at', 'DESC');
            $verified_record = $this->CI->db->get('otp_verification')->row();

            return !empty($verified_record);
        } catch (Exception $e) {
            log_message('error', 'OTP HasVerified Check Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if phone is in cooldown period
     */
    private function isInCooldown($phone, $purpose)
    {
        $cooldown_time = date('Y-m-d H:i:s', strtotime('-60 seconds'));

        $this->CI->db->where('phone', $phone);
        $this->CI->db->where('purpose', $purpose);
        $this->CI->db->where('created_at >', $cooldown_time);

        return $this->CI->db->count_all_results('otp_verification') > 0;
    }

    /**
     * Send OTP SMS using backend style (same as patient module)
     * @param string $phone Phone number
     * @param string $otp_code OTP code
     * @param string $purpose Purpose of OTP
     * @return array Result
     */
    private function sendOTPSMSBackendStyle($phone, $otp_code, $purpose)
    {
        try {
            // Load required models and libraries like backend
            $this->CI->load->model('sms/sms_model');
            $this->CI->load->model('hospital/hospital_model');
            // $this->CI->load->library('sms');

            // Get hospital ID
            $hospital_id = $this->CI->session->userdata('hospital_id') ?: 1;

            // Check if hospital allows SMS (simplified check)
            // $hospital_info = $this->CI->hospital_model->getHospitalById($hospital_id);
            // if (!$hospital_info) {
            //     return array('success' => false, 'message' => 'Hospital information not found');
            // }

            // Create direct OTP message (no template needed)
            $message = "Your verification code is: $otp_code. Valid for 5 minutes. Do not share this code with anyone.";

            // Send SMS using the backend SMS system (same as patient module)
            $data2[] = array($phone => $message);

            // Use the sms library sendSms method (same as backend)
            // $result = $this->CI->sendSms($phone, $message, $data2);
            $result = $this->sendSMS($phone, $message);
            // Log the SMS attempt
            $sms_log_data = array(
                'message' => $message,
                'date' => time(),
                'recipient' => 'OTP - Phone: ' . $phone . ' Purpose: ' . $purpose,
                'user' => $this->CI->ion_auth->get_user_id() ?: 0,
                'hospital_id' => $hospital_id
            );
            $this->CI->sms_model->insertSms($sms_log_data);

            return array('success' => true, 'message' => 'OTP SMS sent successfully with direct message');
        } catch (Exception $e) {
            log_message('error', 'Backend Style OTP SMS Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'Failed to send OTP SMS: ' . $e->getMessage());
        }
    }

    /**
     * Send SMS using configured gateway (legacy method)
     */
    private function sendSMS($phone, $message)
    {
        try {
            // Get SMS gateway settings
            // $sms_gateway = $this->CI->settings_model->getSettings()->sms_gateway;
            // $sms_gateway = $this->CI->settings_model->getSettings(['hospital_id' => 'superadmin'])->sms_gateway;


            // if (empty($sms_gateway)) {
            //     return array('success' => false, 'message' => 'No SMS gateway configured');
            // }
            $hospital_id = 'superadmin';
            $settings = $this->CI->db->get_where('settings', ['hospital_id' => $hospital_id])->row();
            $sms_gateway = $settings ? $settings->sms_gateway : null;

            if (empty($sms_gateway)) {
                return array('success' => false, 'message' => 'No SMS gateway configured for hospital ID ' . $hospital_id);
            }


            $sms_settings = $this->CI->sms_model->getSmsSettingsByGatewayNameBySuperadmin($sms_gateway);

            if (!$sms_settings) {
                return array('success' => false, 'message' => 'SMS gateway settings not found');
            }

            // Send based on gateway type
            $result = $this->sendByGateway($sms_settings, $phone, $message);

            // Log SMS
            $sms_log_data = array(
                'phone' => $phone,
                'message' => $message,
                'gateway' => $sms_gateway,
                'purpose' => 'OTP_VERIFICATION',
                'sent_at' => date('Y-m-d H:i:s'),
                'hospital_id' => 'superadmin',
                'status' => $result['success'] ? 'sent' : 'failed'
            );

            $this->CI->db->insert('sms', $sms_log_data);

            return $result;
        } catch (Exception $e) {
            log_message('error', 'SMS Send Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'Failed to send SMS');
        }
    }

    /**
     * Send SMS using specific gateway
     */
    private function sendByGateway($sms_settings, $phone, $message)
    {
        try {
            if ($sms_settings->name == 'Clickatell') {
                $response = file_get_contents("https://api.clickatell.com/http/sendmsg?user=" . urlencode($sms_settings->username) . "&password=" . urlencode($sms_settings->password) . "&api_id=" . urlencode($sms_settings->api_id) . "&to=" . urlencode($phone) . "&text=" . urlencode($message));
                return array('success' => true, 'message' => 'SMS sent via Clickatell');
            } elseif ($sms_settings->name == 'MSG91') {
                $response = file_get_contents('http://world.msg91.com/api/v2/sendsms?authkey=' . urlencode($sms_settings->authkey) . '&mobiles=' . urlencode($phone) . '&message=' . urlencode($message) . '&sender=' . urlencode($sms_settings->sender) . '&route=4&country=0');
                return array('success' => true, 'message' => 'SMS sent via MSG91');
            } elseif ($sms_settings->name == 'Twilio') {
                require_once APPPATH . 'third_party/twilio/src/Twilio/autoload.php';
                $client = new Twilio\Rest\Client($sms_settings->sid, $sms_settings->token);
                $client->messages->create($phone, array('from' => $sms_settings->sendernumber, 'body' => $message));
                return array('success' => true, 'message' => 'SMS sent via Twilio');
            } elseif ($sms_settings->name == 'Bdbulksms') {
                $data = array('to' => $phone, 'message' => $message, 'token' => $sms_settings->sms_token);
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, "https://api.bdbulksms.net/api.php?json");
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                $response = curl_exec($ch);
                curl_close($ch);
                return array('success' => true, 'message' => 'SMS sent via Bdbulksms');
            } elseif ($sms_settings->name == '80Kobo') {
                $data = array("email" => $sms_settings->email, "password" => $sms_settings->password, "message" => $message, "sender_name" => $sms_settings->sender_name, "recipients" => $phone);
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, 'https://api.80kobosms.com/v2/app/sms');
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
                curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                $response = curl_exec($ch);
                curl_close($ch);
                return array('success' => true, 'message' => 'SMS sent via 80Kobo');
            } elseif ($sms_settings->name == 'Hutch BSMS') {
                $username = $sms_settings->hutch_username; // your Hutch username
                $password = $sms_settings->hutch_password; // your Hutch password
                $mask     = $sms_settings->mask; // pre-approved sender ID
                $number   = $phone;
                $text  = $message;
                $campaign = "Hospital Campaign";
                // 1. Login to get access token
                $loginUrl = "https://bsms.hutch.lk/api/login";
                $loginData = [
                    "username" => $username,
                    "password" => $password
                ];

                $ch = curl_init($loginUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    "Content-Type: application/json",
                    "Accept: */*",
                    "X-API-VERSION: v1"
                ]);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($loginData));
                $loginResponse = curl_exec($ch);
                curl_close($ch);

                $loginResult = json_decode($loginResponse, true);

                if (empty($loginResult['accessToken'])) {
                    echo "Login failed: " . $loginResponse;
                    return;
                }

                $accessToken = $loginResult['accessToken'];

                // 2. Send SMS using access token
                $smsUrl = "https://bsms.hutch.lk/api/sendsms";
                $smsData = [
                    "campaignName" => $campaign,
                    "mask" => $mask,
                    "numbers" => $number,
                    "content" => $text,
                    "deliveryReportRequest" => true
                ];

                $ch = curl_init($smsUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    "Content-Type: application/json",
                    "Accept: */*",
                    "X-API-VERSION: v1",
                    "Authorization: Bearer " . $accessToken
                ]);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($smsData));
                $smsResponse = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                return array('success' => true, 'message' => 'SMS sent successfully');
            } elseif ($sms_settings->name == 'Dialog') {


                error_reporting(E_ALL);
                date_default_timezone_set('Asia/Colombo');

                $now = date("Y-m-d\TH:i:s");
                $username = $sms_settings->dialog_username;
                $password = $sms_settings->dialog_password;
                $digest = md5($password);
                $number = $phone;
                $mask = $sms_settings->dialog_mask;
                $text = $message;
                $campaignName = 'Hospital SMS Campaign';
                $clientRef = str_pad(mt_rand(0, 9999999), 7, '0', STR_PAD_LEFT);

                // Build the payload as a PHP array
                $bodyArray = [
                    "messages" => [
                        [
                            "clientRef" => $clientRef,
                            "number" => $number,
                            "mask" => $mask,
                            "text" => $text,
                            "campaignName" => $campaignName
                        ]
                    ]
                ];

                // Convert array to JSON
                $body = json_encode($bodyArray);

                // Initialize CURL
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, "https://richcommunication.dialog.lk/api/sms/send");
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                // Set headers
                $headers = [
                    'Content-Type: application/json',
                    'USER: ' . $username,
                    'DIGEST: ' . $digest,
                    'CREATED: ' . $now
                ];
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

                // Execute request
                $server_output = curl_exec($ch);
                $curl_error = curl_error($ch);
                curl_close($ch);
                $response = json_decode($server_output, true);
                return array('success' => true, 'message' => 'SMS sent successfully');
            }
        } catch (Exception $e) {
            log_message('error', 'SMS Gateway Error: ' . $e->getMessage());
            return array('success' => false, 'message' => 'Gateway communication error');
        }
    }
}
